function [ x, istop, itn, rnorm, anorm, acond, arnorm, xnorm, var, cov ]...
  = covlsqr( m, n, Aprod, b, damp, atol, btol, conlim, itnlim, show, IC )

% [ x, istop, itn, r1norm, r2norm, anorm, acond, arnorm, xnorm, var, cov ]...
%  = covlsqr( m, n, Aprod, b, damp, atol, btol, conlim, itnlim, show, IC )

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%matlab%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% File             :  covlsqr.m
% Description      :  This is a modified version of the free software LSQR 
%                     by C. C. Paige and M. A. Saunders (for download under 
%                     http://www.stanford.edu/group/SOL/software.html).
%                     covlsqr.m (based on LSQR) is a solver for sparse 
%                     linear equations and sparse least square problems;
%                     covlsqr.m can also estimate the parameter covariance
%                     matrix for a large parameter estimation problem with
%                     nonlinear equality constraints.
% Author           :  Inga Schierle
% email            :  inga.schierle@gmx.de
% Created on       :  May 28, 2007
% Last modified by :  Michael Saunders
% Last modified on :  24 Dec 2010
% Update count     :  1
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Date          Name           Changes / Extensions
% ----          ----           --------------------
% 24 Dec 2010   MAS            Reverse inputs for Aprod.  
%                              Move cov output to end.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%--------------------------------------------------------------------------
% covlsqr is an updated implementation of lsqr.m from SOL, Stanford University.
% It allows A to be a function handle, and it can return estimates
% of any principal submatrix of the covariance matrix.
%
% LSQR solves  Ax = b  or  min ||b - Ax||_2  if damp = 0,
% or   min || (b)  -  (  A   )x ||   otherwise.
%          || (0)     (damp I)  ||2
% A  is an m by n matrix defined by  y = Aprod( v, mode )
% where 'Aprod' is a function handle that performs the matrix-vector operations.
% If mode = 1,   Aprod returns y = Ax.
% If mode = 2,   Aprod returns y = A'x.
%--------------------------------------------------------------------------
% LSQR uses an iterative (conjugate-gradient-like) method.
% For further information, see 
% 1. C. C. Paige and M. A. Saunders (1982a).
%    LSQR: An algorithm for sparse linear equations and sparse least squares,
%    ACM TOMS 8(1), 43-71.
% 2. C. C. Paige and M. A. Saunders (1982b).
%    Algorithm 583.  LSQR: Sparse linear equations and least squares problems,
%    ACM TOMS 8(2), 195-209.
% 3. M. A. Saunders (1995).  Solution of sparse rectangular systems using
%    LSQR and CRAIG, BIT 35, 588-604.
%
% Input parameters:
% Aprod       is a function handle that defines A. 
%             y = Aprod(v,1) should return y = A*v.
%             y = Aprod(v,2) should return y = A'*v.
% atol, btol  are stopping tolerances.  If both are 1.0e-9 (say),
%             the final residual norm should be accurate to about 9 digits.
%             (The final x will usually have fewer correct digits,
%             depending on cond(A) and the size of damp.)
% conlim      is also a stopping tolerance.  lsqr terminates if an estimate
%             of cond(A) exceeds conlim.  For compatible systems Ax = b,
%             conlim could be as large as 1.0e+12 (say).  For least-squares
%             problems, conlim should be less than 1.0e+8.
%             Maximum precision can be obtained by setting
%             atol = btol = conlim = zero, but the number of iterations
%             may then be excessive.
% itnlim      is an explicit limit on iterations (for safety).
% show = 1    gives an iteration log,
% show = 0    suppresses output.
% IC          is an index set specifying which rows and columns of
%             the covariance matrix COV = inv(A'*A + damp^2 I) should
%             be estimated.  For example, if IC = [2 4], covlsqr
%             will estimate the principal submatrix that contains
%             rows [2 4] and columns [2 4] of COV.  That is, it will
%             return the estimate cov ~= COV(IC,IC).
%
% Output parameters:
% x           is the final solution.
% istop       gives the reason for termination.
% istop       = 1 means x is an approximate solution to Ax = b.
%             = 2 means x approximately solves the least-squares problem.
% rnorm       = sqrt( norm(r)^2  +  damp^2 * norm(x)^2 ).
% anorm       = estimate of Frobenius norm of Abar = [  A   ].
%                                                    [damp*I]
% acond       = estimate of cond(Abar).
% arnorm      = estimate of norm(A'*r - damp^2*x).
% xnorm       = norm(x).
% var         (if present) estimates all diagonals of (A'A)^{-1} (if damp=0)
%             or more generally (A'A + damp^2*I)^{-1}.
%             This is well defined if A has full column rank or damp > 0.
%             (Not sure what var means if rank(A) < n and damp = 0.)
% cov         Principal submatrix COV(IC,IC) of the parameter covariance COV
%             for the considered parameter estimation problem.
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
%     Initialize.
%--------------------------------------------------------------------------

msg=['The exact solution is  x = 0                              '
     'Ax - b is small enough, given atol, btol                  '
     'The least-squares solution is good enough, given atol     '
     'The estimate of cond(Abar) has exceeded conlim            '
     'Ax - b is small enough for this machine                   '
     'The least-squares solution is good enough for this machine'
     'Cond(Abar) seems to be too large for this machine         '
     'The iteration limit has been reached                      '];

var = zeros(n,1);
t   = length(IC);
cov = zeros(t,t);
 
if show
   disp(' ')
   disp('LSQR            Least-squares solution of  Ax = b')
   str1 = sprintf('The matrix A=R has %8g rows  and %8g cols', m, n);
   str2 = sprintf('damp = %20.14e ', damp);
   str3 = sprintf('atol = %8.2e                 conlim = %8.2e', atol, conlim);
   str4 = sprintf('btol = %8.2e                 itnlim = %8g'  , btol, itnlim);
   disp(str1);   disp(str2);   disp(str3);   disp(str4);
end

itn    = 0;		
istop  = 0;		
nstop  = 0;
ctol   = 0;		
if conlim > 0, 
    ctol = 1/conlim; 
end;
anorm  = 0;
acond  = 0;
dampsq = damp^2;
ddnorm = 0;
res2   = 0;
xnorm  = 0;
xxnorm = 0;
z      = 0;
cs2    = -1;
sn2    = 0;

%--------------------------------------------------------------------------
% Set up the first vectors u and v for the bidiagonalization.
% These satisfy  beta*u = b,  alfa*v = A'u.
%--------------------------------------------------------------------------

u    = b(1:m);
x    = zeros(n,1);
alfa = 0;
beta = norm(u);
if beta > 0
   u    = (1/beta)*u;	
   v    = Aprod(u,2);
   alfa = norm(v);
end
if alfa > 0
   v    = (1/alfa)*v;    
   w    = v;
end

arnorm = alfa*beta;   
if arnorm==0 
    disp(msg(1,:)) 
    return 
end

rhobar = alfa;		
phibar = beta;		
bnorm  = beta;
rnorm  = beta;
head1  = '   Itn      x(1)      rnorm   ';
head2  = 'Compatible   LS      Norm A   Cond A';

if show
   disp(' ')
   disp([head1 head2])
   test1  = 1;		
   test2  = alfa/beta;
   fprintf('%6g %12.5e %10.3e %8.1e %8.1e', itn, x(1), rnorm, test1, test2)
end

%--------------------------------------------------------------------------
%     Main iteration loop.
%--------------------------------------------------------------------------

while itn < itnlim
      itn = itn + 1;
      
%--------------------------------------------------------------------------      
%     Perform the next step of the bidiagonalization to obtain the
%     next  beta, u, alfa, v.  These satisfy the relations
%                beta*u  =  a*v   -  alfa*u,
%                alfa*v  =  A'*u  -  beta*v.
%--------------------------------------------------------------------------
      
      u = Aprod(v,1) - alfa*u;
      beta = norm(u);
      if beta > 0
         u     = (1/beta)*u;
         anorm = norm([ anorm alfa beta damp]);
         v     = Aprod(u,2 ) - beta*v;
         alfa  = norm(v);
         if alfa > 0  
             v = (1/alfa)*v;
         end
      end

%--------------------------------------------------------------------------
%     Use a plane rotation to eliminate the damping parameter.
%     This alters the diagonal (rhobar) of the lower-bidiagonal matrix.
%--------------------------------------------------------------------------

      rhobar1 = norm([rhobar damp]);
      cs1     = rhobar/rhobar1;
      sn1     = damp  /rhobar1;
      psi     = sn1*phibar;
      phibar  = cs1*phibar;

%--------------------------------------------------------------------------
%     Use a plane rotation to eliminate the subdiagonal element (beta)
%     of the lower-bidiagonal matrix, giving an upper-bidiagonal matrix.
%--------------------------------------------------------------------------

      rho     =   norm([rhobar1 beta]);
      cs      =   rhobar1/rho;
      sn      =   beta   /rho;
      theta   =   sn*alfa;
      rhobar  = - cs*alfa;
      phi     =   cs*phibar;
      phibar  =   sn*phibar;
      tau     =   sn*phi;

%--------------------------------------------------------------------------
%     Update x, w, b, var and cov.
%--------------------------------------------------------------------------

      t1      =   phi  /rho;
      t2      = - theta/rho;
      dk      =   (1/rho)*w;
      var     =   var + dk.*dk;
      
      if IC ~= 0
          cov = cov + dk(IC)*dk(IC)';
      end

      x       = x      + t1*w;
      w       = v      + t2*w;
      ddnorm  = ddnorm + norm(dk)^2;
      
%--------------------------------------------------------------------------
%     Use a plane rotation on the right to eliminate the
%     super-diagonal element (theta) of the upper-bidiagonal matrix.
%     Then use the result to estimate  norm(x).
%--------------------------------------------------------------------------

      delta   =   sn2*rho;
      gambar  = - cs2*rho;
      rhs     =   phi - delta*z;
      zbar    =   rhs/gambar;
      xnorm   =   sqrt(xxnorm + zbar^2);
      gamma   =   norm([gambar theta]);
      cs2     =   gambar/gamma;
      sn2     =   theta /gamma;
      z       =   rhs   /gamma;
      xxnorm  =   xxnorm + z^2;

%--------------------------------------------------------------------------
%     Test for convergence.
%     First, estimate the condition of the matrix  Abar,
%     and the norms of  rbar  and  Abar'rbar.
%--------------------------------------------------------------------------

      acond   =   anorm*sqrt(ddnorm);
      res1    =   phibar^2;
      res2    =   res2 + psi^2;
      rnorm   =   sqrt(res1 + res2);
      arnorm  =   alfa*abs(tau);

%--------------------------------------------------------------------------
%     Now use these norms to estimate certain other quantities,
%     some of which will be small near a solution.
%--------------------------------------------------------------------------

      test1   =   rnorm / bnorm;
      test2   =   arnorm/(anorm*rnorm);
      test3   =   1     / acond;
      t1      =   test1 /(1 +  anorm*xnorm/bnorm);
      rtol    =   btol  + atol*anorm*xnorm/bnorm;

%--------------------------------------------------------------------------
%     The following tests guard against extremely small values of
%     atol, btol  or  ctol.  (The user may have set any or all of
%     the parameters  atol, btol, conlim  to 0.)
%     The effect is equivalent to the normal tests using
%     atol = eps,  btol = eps,  conlim = 1/eps.
%--------------------------------------------------------------------------

      if itn >= itnlim,   istop = 7; end
      if 1 + test3  <= 1, istop = 6; end
      if 1 + test2  <= 1, istop = 5; end
      if 1 + t1     <= 1, istop = 4; end

%     Allow for tolerances set by the user.

      if  test3 <= ctol,  istop = 3; end
      if  test2 <= atol,  istop = 2; end
      if  test1 <= rtol,  istop = 1; end

%     See if it is time to print something.

      prnt = 0;
      if n     <= 40        , prnt = 1; end
      if itn   <= 10        , prnt = 1; end
      if itn   >= itnlim-10 , prnt = 1; end
      if rem( itn, 10 ) == 0, prnt = 1; end
      if test3 <=  2*ctol   , prnt = 1; end
      if test2 <= 10*atol   , prnt = 1; end
      if test1 <= 10*rtol   , prnt = 1; end
      if istop ~=  0        , prnt = 1; end

      if prnt == 1
         if show
            fprintf('\n%6g %12.5e %10.3e %8.1e %8.1e %8.1e %8.1e', ...
                    itn, x(1), rnorm, test1, test2, anorm, acond);
         end
      end
      if istop > 0
          break
      end
end

%--------------------------------------------------------------------------
%     End of iteration loop.
%     Print the stopping condition.
%--------------------------------------------------------------------------

if show
   fprintf('\n\ncovlsqr finished\n')
   disp(msg(istop+1,:))
   disp(' ')
   str1 = sprintf('istop =%8g   rnorm =%8.1e',    istop, rnorm );
   str2 = sprintf('anorm =%8.1e   arnorm =%8.1e', anorm, arnorm);
   str3 = sprintf('itn   =%8g               ',    itn          );
   str4 = sprintf('acond =%8.1e   xnorm  =%8.1e', acond, xnorm );
   disp([str1 '   ' str2])
   disp([str3 '   ' str4])
   disp(' ')
end

%-----------------------------------------------------------------------
% End of covlsqr.m
%-----------------------------------------------------------------------
