function [dx, cov] = mytestLSQR

%        [dx2, cov2] = mytestLSQR
% solves a constrained least-squares problem
%    min 1/2 || J1*dx + b1 ||^2
%    s.t.       J2*dx + b2 = 0
% (defined by myset.m) using the iterative solver covLSQR.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%matlab%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% File             :  mytestLSQR.m
% Description      :  Solver (based on LSQR) for a linear least 
%                     square problem with linear constraints;
%                     mytestLSQR.m can also compute the parameter covariance
%                     matrix for a large parameter estimation problem with
%                     nonlinear equality constraints.
% Author           :  Inga Schierle
% email            :  inga.schierle@gmx.de
% Created on       :  May 28, 2007
% Last modified by :  Michael Saunders
% Last modified on :  24 Dec 2010
% Update count     :  1
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Date          Name           Changes / Extensions
% ----          ----           --------------------
% 24 Dec 2010   MAS            Follow notation in paper.
%                              Reverse inputs for Aprod.
%                              Move cov output to end.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%--------------------------------------------------------------------------
% To solve
%    min 1/2 || J1*dx + b1 ||^2
%    s.t.       J2*dx + b2 = 0
% we first solve the unconstrained problem
%    min || (J1*Z)*w - g ||^2 (= min || J1*(Z*Z')*s - g||^2)
% using LSQR.
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Definition of J1, J2, f1, f2, and IC.
%--------------------------------------------------------------------------
[ m1, m2, n, J1, J2, f1, f2, IC ] = myset;
n2 = n - m2;

%--------------------------------------------------------------------------
% Initialization of some parameters
%--------------------------------------------------------------------------
damp   = 0;
atol   = 1.0e-10;
btol   = 1.0e-10;
conlim = 1.0e+8;
itnlim = 15000;
show   = 1;
show2  = 0;

%--------------------------------------------------------------------------
% Computation of a preconditioner D2 for the problem min || J2'q - v || for
% given vectors v and a preconditioner for the problem J2*x = -f2.
%--------------------------------------------------------------------------
% d2 = col2norms of J2' (= row2norms of J2)
d2 = zeros( m2, 1 );

for i = 1:m2
    d2(i) = norm( J2( i, : ), 2 );
end

d2 = 1./d2;
D2 = sparse( 1:m2, 1:m2, d2 );

%--------------------------------------------------------------------------
% Definition of some handle functions
%--------------------------------------------------------------------------
J2TD2 = J2' * D2;

% For estimating the colnorms of J2'
% colnorms(J2',n,m2,20);

J1prod      = @(v,mode ) aprod( v,  mode, J1 );
J2prod      = @(v,mode ) aprod( v,  mode, J2 );
J2Tprod     = @(v,mode ) aprod( v,3-mode, J2 );
J2TDprod    = @(v,mode ) aprod( v,  mode, J2TD2 );
DJ2prod     = @(v,mode ) aprod( v,3-mode, J2TD2 );
J1ZZTprod   = @(v,mode ) J1ZZTproduct( v, mode, J1prod, J2TDprod, n, m2 );

%--------------------------------------------------------------------------
% Computation of g using left-preconditioning and LSQR.
% Apply LSQR to J2*x = -f2
%--------------------------------------------------------------------------
df2     = d2 .* f2;
df2norm = norm(df2,inf);
df2     = df2 / df2norm;
y       = covlsqr( m2, n, DJ2prod, -df2, damp, atol, btol, conlim, itnlim, show2, 0 );
y       = y * df2norm;
g       = - f1 - J1*y;

%--------------------------------------------------------------------------
% solve min || (J1*Z*Z^T)*s - g ||^2 with LSQR
%--------------------------------------------------------------------------
[ s, istop, itn, rnorm, anorm, acond, arnorm, xnorm, var, cov ] ...
    = covlsqr(m1, n, J1ZZTprod, g, damp, atol, btol, conlim, itnlim, show2, IC );

%--------------------------------------------------------------------------
% Computation of x = Z*w + dx.
%--------------------------------------------------------------------------
% disp('solution for Z*w using LSQR');
sol1 = covlsqr( n, m2, J2TDprod, s, damp, atol, btol, conlim, itnlim, show2, 0 );
Zw   = s - J2TDprod(sol1,1);
dx    = Zw + y;

%--------------------------------------------------------------------------
% end of mytestLSQR.m
%--------------------------------------------------------------------------
