from math import sqrt, log

import numpy as np

from scipy.sparse.linalg import LinearOperator, aslinearoperator

from mpi4py import MPI

def ls_nesterov( A, b, over, s_max, s_min, tol = 1e-8, comm = MPI.COMM_WORLD ):

    A = aslinearoperator(A)
    m, n = A.shape

    L  = s_max**2
    mu = s_min**2

    alpha = 4.0/(3.0*L+mu)
    beta  = (1.0-sqrt(alpha*mu))/(1.0+sqrt(alpha*mu))
    rate  = 1.0-sqrt(alpha*mu)

    eps = np.finfo(float).eps
    if tol < eps:
        tol = eps
        
    iter_lim = np.ceil(log(tol)/log(rate))

    if over is True:                    # over-determined

        x = np.zeros(n)
        y = np.zeros(n)

        Atb = comm.allreduce(A.rmatvec(b))

        for itn in xrange(int(iter_lim)):
            x_p  = x
            Ay   = A.matvec(y)
            x    = y - alpha*(comm.allreduce(A.rmatvec(Ay))-Atb)
            y    = (1+beta)*x - beta*x_p

    else:                               # under-determined

        x = np.zeros(n)
        y = np.zeros(n)

        Atb = A.rmatvec(b)

        for itn in xrange(int(iter_lim)):
            x_p = x
            Ay  = comm.allreduce(A.matvec(y))
            x   = y  - alpha*(A.rmatvec(Ay)-Atb)
            y   = (1+beta)*x - beta*x_p

    return y, iter_lim
            
