from math import sqrt, log

import numpy as np
from numpy.linalg import lstsq, norm

from scipy.sparse.linalg import LinearOperator, aslinearoperator

def ls_nesterov( A, b, s_max, s_min, tol = 1e-8, iter_lim = None ):
    """
    Nesterov's method for linear least squares problems
    
    INPUTS:
    -------

    A : (m,n) ndarray, matrix, or LinearOperator

    b : (m,) ndarray
    
    s_max : float, an upper bound of the largest singular value of A

    s_min : float, a lower bound of the smallest non-zero singular value of A

    tol : float, tolerance on relative error

    iter_lim : int, iteration limit

    OUTPUTS:
    --------

    x : (n,) ndarray, the approximate solution

    flag : int, 0 : approximately solved, 1 : [s_min, s_max] doesn't bound A's singular values
    
    itn : int, number of iterations
    """

    A    = aslinearoperator(A)
    m, n = A.shape

    def AtA_op_matvec(v):
        return A.rmatvec(A.matvec(v))
    AtA = LinearOperator( (n,n), matvec = AtA_op_matvec )
    Atb = A.rmatvec(b)

    x, flag, itn = quad_nesterov( AtA, Atb, s_max**2, s_min**2, tol, iter_lim )

    return x, flag, itn

def quad_nesterov( A, b, L, mu, tol=1e-8, iter_lim = None ):
    """
    Nesterov's method for symmetric and positive semi-definite linear systems or linear
    least squares problems.

    INPUTS:
    -------
    
    A : (n,n) ndarray, matrix, LinearOperator

    b : (n,) ndarray
    
    L : float, an upper bound of the largest eigenvalue of A

    mu : float, a lower bound of the smallest non-zero eigenvalue of A

    tol : float, tolerance on relative error

    iter_lim : int, iteration limit

    OUTPUTS:
    --------

    x : (n,) ndarray, the approximate solution

    flag : int, 0 : approximately solved, 1 : [mu, L] doesn't bound A's eigenvalues
    
    itn : int, number of iterations
    """

    A    = aslinearoperator(A)
    n    = A.shape[0]

    alpha = 4.0/(3.0*L+mu)
    beta  = (1.0-sqrt(alpha*mu))/(1.0+sqrt(alpha*mu))
    rate  = 1.0-sqrt(alpha*mu)

    maxit = np.ceil(log(tol)/log(rate))
    if (iter_lim is None) or (iter_lim > maxit):
        iter_lim = maxit

    x    = np.zeros(n)
    y    = np.zeros(n)

    for itn in xrange(int(iter_lim)):
        x_p = x
        x   = y - alpha*(A.matvec(y)-b)
        y   = (1+beta)*x - beta*x_p

    if norm(x) <= norm(b)/mu:
        flag = 0
    else:
        flag = 1

    return y, flag, iter_lim

def _test():

    m      = 100
    n      = 200
    A      = np.random.randn(m,n)
    s_max  = sqrt(n)+sqrt(m)
    s_min  = sqrt(n)-sqrt(m)
    b      = np.random.randn(m)
    x_opt, = lstsq(A,b)[:1]

    tol          = 1e-12
    x, flag, itn = ls_nesterov(A,b,s_max,s_min,tol)[:3]

    relerr = norm(x-x_opt)/norm(x_opt)

    print relerr, flag, itn

if __name__ == '__main__':
    _test()
    
