
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "commonlib.h"
#include "myblas.h"
#include "lumod_dense.h"
#include "LUMOD_D.h"

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

void lutest ( int m, int n, int nrowb, int ncolb, int mtest,
              REAL r1, REAL r2, REAL densty, int *seeds,
              REAL *B, REAL *C, REAL *D, REAL *L, REAL *U, REAL *v, REAL *w,
              REAL *x, REAL *y, REAL *z, int verbose )
{

/*      REAL precision B[nrowb,ncolb], C[m,m], D[m],
                         L[*], U[*], v[m], w[m],
                         x[m], y[m], z[m];
*/
/* ------------------------------------------------------------------
   LUTEST is a test program for LUMOD.
   ------------------------------------------------------------------ */

  REAL stats[7+BLAS_BASE];
  REAL zero = 0.0;
  REAL big  = BIGNUMBER;

  int  ntest, i, irep, inew, nn, k, j, jrep, jnew, kdel, krow, kcol;
  REAL emax, time;

/*
  Initialize stats[*].
*/

  stats[1] = zero;  // Cmax = max |Cij|
  stats[2] = zero;  // Lmax = max |Lij|
  stats[3] = zero;  // Umax = max |Uij|
  stats[4] = big;   // Umin = min |Ujj|
  stats[5] = zero;  // Fmax = max |L*C - U|
  stats[6] = zero;  // smax = |Cx - b|
  stats[7] = zero;  // tmax = |C'y - b|

  krow = 0;
  kcol = 0;

/* Run the specified number of tests. */

  for (ntest = 1; ntest<=mtest; ntest++) {
	 if(verbose) {
     printf("\n");
     printf("\n");
	   printf("------------------------------------------\n");
     printf("%s %4d%s %6d\n", " LUTEST", ntest, ".      m =", m);
	   printf("------------------------------------------\n");
	 }
	 else
	   printf("\nTest %d (%dx%d)", ntest, n,m);

     setmat( m, n, nrowb, &B[submat(nrowb,  1,  1)], D, v, w,
             r1, r2, densty, seeds, y );
     setmat( m, n, nrowb, &B[submat(nrowb,  1,n+1)], D, v, w,
            r1, r2, densty, seeds, y );
     setmat( m, n, nrowb, &B[submat(nrowb,m+1,  1)], D, v, w,
             r1, r2, densty, seeds, y );

/*   ---------------------------------------------------------------
     Print the matrix.
     --------------------------------------------------------------- */
/*
	 for (nn = 1; nn<=nrowb; nn++) {
		for (k = 1; k<=ncolb; k++)
          printf("%14g", B[posmat(nrowb, nn,k)]);
       printf("\n");
	 }
*/

/*   ---------------------------------------------------------------
     Test mode 1, adding 1 row and a column at a time.
     --------------------------------------------------------------- */
   time = timeNow();
	 if(verbose)
       printf("\nLUmod (mode 1):\n");
	 else
	   printf("\n1");

     for (nn = 1; nn<=m; nn++) {
       for (k = 1; k<=nn; k++) {
         y[k]  = B[posmat(nrowb,nn,k)];
         z[k]  = B[posmat(nrowb,k,nn)];
       }
       dcopy ( nn, y, 1, &C[submat(n, nn,1)], m );
       dcopy ( nn, z, 1, &C[submat(n, 1,nn)], 1 );

       LUmod ( 1, m, nn, krow, kcol, L, U, y, z, w );

		   if(verbose)
         printf("%s %d %s\n", " Row and column", nn, "added");
       else
	       printf(".");

		   luchek( m, n, nn, C, L, U, stats, v, w, x, y, verbose );
     }
//goto x4;
//goto Done;

/*   ---------------------------------------------------------------
     Replace columns of the LU one by one.
     Old ones are replaced by new ones with no reordering.
     --------------------------------------------------------------- */
x2:
     if(verbose)
       printf("\nLUmod (mode 2):\n");
	   else
	     printf("\n2");

     for (j = 1; j<=n; j++) {
       jrep  = j;
       jnew  = n + j;

       for (i = 1; i<=m; i++)
         z[i]  = B[posmat(nrowb,i,jnew)];

       dcopy ( m, z, 1, &C[submat(n,1,jrep)], 1 );

       LUmod ( 2, m, m, krow, jrep, L, U, y, z, w );

		   if(verbose)
         printf("%s %d %s %d\n", "Column", jrep, "replaced by column", jnew);
	     else
	       printf(".");

       luchek( m, n, m, C, L, U, stats, v, w, x, y, verbose );
     }
//goto x4;
//goto Done;

/*   ---------------------------------------------------------------
     Replace rows of the LU one by one.
     Old ones are replaced by new ones with no reordering.
     --------------------------------------------------------------- */
x3:
	   if(verbose)
       printf("\nLUmod (mode 3):\n");
	   else
	     printf("\n3");

     for (i = 1; i<=m; i++) {
       irep  = i;
       inew  = m + i;

       for (j = 1; j<=n; j++)
         y[j]  = B[posmat(nrowb,inew,j)];

       dcopy ( n, y, 1, &C[submat(n,irep,1)], m );

       LUmod ( 3, m, m, irep, kcol, L, U, y, z, w );

		   if(verbose)
         printf("%s %d %s %d\n", "Row", irep, "replaced by row", inew);
	     else
	       printf(".");

       luchek( m, n, m, C, L, U, stats, v, w, x, y, verbose );
     }

/*   ---------------------------------------------------------------
     Delete a specified row and column of the LU one by one,
     replacing them by the last row and column.
     --------------------------------------------------------------- */
x4:
     if(verbose)
       printf("\nLUmod (mode 4):\n");
	   else
	     printf("\n4");

     nn     = m;
     for (kdel = 1; kdel<m; kdel++) {
       krow  = max( nn/2, 1 );
       kcol  = krow;

       dcopy ( nn  , &C[submat(n,1,nn)], 1, &C[submat(n,1,kcol)], 1 );
       dcopy ( nn-1, &C[submat(n,nn,1)], m, &C[submat(n,krow,1)], m );

       LUmod ( 4, m, nn, krow, kcol, L, U, y, z, w );
       nn--;

		   if(verbose)
         printf("%s %d %s %d %s\n", "Row", krow, "and column", kcol, "deleted");
	     else
	       printf(".");

       luchek( m, n, nn, C, L, U, stats, v, w, x, y, verbose );
     }
  }

Done:
  time = timeNow()-time;
  emax   = max(stats[5], stats[6] );
  emax   = max(stats[7], emax );

  printf("\n\n");
//  printf("Aggregate statistics for %d randomized LU tests\n", mtest);
  printf("Data range---------------------------Density %5.1f%%\n", (densty*densty*100));
  printf("    max|Cij|     max|Lij|     max|Uij|     min|Ujj|\n");
  for (j = 1; j<= 4; j++)
    printf("%12g ", stats[j]);
  printf("\n");
  printf("Solver accuracy------------------------------------\n");
  printf("   max|LC-U|    max|Cx-b|   max|C'y-b|       STATUS\n");
  for (j = 5; j<= 7; j++)
    printf("%12g ", stats[j]);
  if (emax <= ERRLIMIT)
     printf("%12s", "OK!");
  else
     printf("%12s", "Bug?");
  printf("\n");
  printf("---------------------------------------------------\n");
  printf("Time in seconds: %.3f\n", time);

}



/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

void setmat( int m, int n, int nrowb, REAL *B, REAL *D, REAL *v, REAL *w,
             REAL r1, REAL r2, REAL densty, int *seeds, REAL *y )
{

  REAL dr1, dr2, ddnsty, wj;
  int    i, j, k;

/* ------------------------------------------------------------------
   setmat sets   B  =  D  +  vw',
   where D = diag(d) and d, v, w are random vectors.
   ------------------------------------------------------------------ */

  dr1    = (r2 - r1) + 1.0;
  dr2    = dr1       + 1.0;
  ddnsty = 1.0;
  randomdens( n, D, dr1, dr2, ddnsty, seeds );
  randomdens( n, v,  r1,  r2, densty, seeds );
  randomdens( n, w,  r1,  r2, densty, seeds );

  for (j = 1; j<=n; j++) {
     wj   = w[j];
	 for (i = 1; i<=m; i++) {
        k = posmat(nrowb,i,j);
        B[k] = v[i] * wj;
	 }
	 k = posmat(nrowb,j,j);
     B[k]+= D[j];
  }
}


/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

void luchek( int m, int n, int nn, REAL *C, REAL *L, REAL *U,
             REAL *stats, REAL *v, REAL *w, REAL *x, REAL *y, int verbose )
{

/* ------------------------------------------------------------------
   LUCHEK computes L*C and compares it with U.
   ------------------------------------------------------------------ */


  REAL zero = 0.0;
  REAL one  = 1.0;
  REAL big  = BIGNUMBER;

  char   *mark;
  int    k, i, j, ll, lu, incl, incu, numu, icmax, imax, ilmax, iumax;
  REAL Cmax, Lmax, Umax, Umin, Fmax, smax, tmax, err;

/* ------------------------------------------------------------------
   Get Cmax, Lmax, Umax, Umin.
   Prevent divide by zero in Usolve -- make sure U[k,k] != 0
   ------------------------------------------------------------------ */

  Cmax   = zero;
  Lmax   = zero;
  Umax   = zero;
  Umin   = big;
  ll     = 1;
  lu     = 1;
  incl   = m;
  incu   = m;
  numu   = nn;

  for (k = 1; k<=nn; k++) {
     icmax  = idamax( nn  , &C[submat(n,1,k)], 1 );
     ilmax  = idamax( nn  , &L[subvec(ll)], 1 );
     iumax  = idamax( numu, &U[subvec(lu)], 1 );
     Cmax   = max( Cmax, fabs(C[posmat(n,icmax,k)]) );
     Lmax   = max( Lmax, fabs(L[ilmax])   );
     Umax   = max( Umax, fabs(U[iumax])   );
     Umin   = min( Umin, fabs(U[lu])      );
     if (U[lu] == zero)
	   U[lu] = MATHPREC;
     ll  += incl;
     lu  += incu;
     incu--;
     numu--;
  }

/* ------------------------------------------------------------------
   Check L*C = U by columns.
   ------------------------------------------------------------------ */

  Fmax   = zero;

  for (k = 1; k<=nn; k++) {

/*   Set  v = L * k-th column of C. */

     Lprod ( 1, m, nn, L, &C[submat(n,1,k)], v );

/*   Find the maximum deviation from the corresponding column of U. */

     dload ( nn, zero, w, 1 );
     lu    = k;
     incu  = m - 1;
     for (i = 1; i<=k; i++) {
        w[i] = U[lu];
        lu  += incu;
        incu--;
     }

     for (i = 1; i<=nn; i++) {
        err  = fabs( w[i] - v[i] );
        Fmax = max( Fmax, err );
     }
  }

/* ------------------------------------------------------------------
   Check solution of C*x = v.
   ------------------------------------------------------------------ */

  x[1]   = one;
  for (j = 2; j<=nn; j++)
     x[j]  = - x[j-1] / 2.0;

  dload ( nn, zero, v, 1 );

  for (k = 1; k<=nn; k++)
     daxpy ( nn, x[k], &C[submat(n,1,k)], 1, v, 1 );

/* Save v in y.
   Solve  C*x = v = y,
   { compute the residual  y - C*x. */

  dcopy ( nn, v, 1, y, 1 );
  Lprod ( 1, m, nn, L, v, x );
/*
if(nn==2) {
       ll = 0;
       for(i = 1; i<=nn; i++) {
         printvec(nn, &L[ll], 0);
         ll += m;
       }  
}
if(nn==2) printvec(nn, x, 0);
*/
  Usolve( 1, m, nn, U, x );
/*
if(nn==2) {
       ll = 0;
       for(i = 1; i<=nn; i++) {
         printvec(nn-i+1, &U[ll], 0);
         ll += m-i+1;
       }  
}
if(nn==2) printvec(nn, x, 0);
*/
  for (k = 1; k<=nn; k++)
     daxpy ( nn, -x[k], &C[submat(n,1,k)], 1, y, 1 );
//if(nn==2) printvec(nn, y, 0);

  imax   = idamax( nn, y, 1 );
  smax   = fabs  ( y[imax] );

/* ------------------------------------------------------------------
   Check solution of C[t]*y = w.
   ------------------------------------------------------------------ */

  y[1]   = one;
  for (i = 2; i<=nn; i++)
     y[i]  = - y[i-1] / 2.0;

  for (k = 1; k<=nn; k++)
     w[k]  = ddot  ( nn, &C[submat(n,1,k)], 1, y, 1 );

/* Save w in x.
   Solve  C[t]*y = w = x,
   { compute the residual  x - C[t]*y. */

  dcopy ( nn, w, 1, x, 1 );
  Usolve( 2, m, nn, U, w );
  Lprod ( 2, m, nn, L, w, y );

  for (k = 1; k<=nn; k++)
     x[k] -= ddot ( nn, &C[submat(n,1,k)], 1, y, 1 );

  imax   = idamax( nn, x, 1 );
  tmax   = fabs  ( x[imax] );

/* Print stats for this matrix and save max or min so far. */

  if(verbose) {
    mark   = " ";
    if ( max(smax,tmax) > TINYNUMBER )
      mark = "*";
/*    printf("   n   max|Cij|   max|Lij|   max|Uij|   min|Ujj|  max|LC-U|  max|Cx-b| max|C'y-b|\n");
    printf("%4d %10g %10g %10g %10g %10g %10g %10g%s\n",
	       nn, Cmax, Lmax, Umax, Umin, Fmax, smax, tmax, mark); */
    printf("   n   max|Lij|   max|Uij|   min|Ujj|  max|LC-U|  max|Cx-b| max|C'y-b|\n");
    printf("%4d %10g %10g %10g %10g %10g %10g%s\n",
	       nn, Lmax, Umax, Umin, Fmax, smax, tmax, mark);
  }
/*  else
    printf("."); */

  stats[1] = max( stats[1], Cmax );
  stats[2] = max( stats[2], Lmax );
  stats[3] = max( stats[3], Umax );
  stats[4] = min( stats[4], Umin );
  stats[5] = max( stats[5], Fmax );
  stats[6] = max( stats[6], smax );
  stats[7] = max( stats[7], tmax );

}



/* ************************************************************************ */

void main( int argc, char *argv[], char *envp[] )
{

/* ------------------------------------------------------------------
   Test program for LUMOD.

   25 Apr 1990:  First version, derived from test program for QRMOD.
   LUTEST is asked to do mtest tests, each of which maintains the
   factors of C as various modes of LUMOD are exercised.
   C builds up to a maximum dimension of maxmod.
   To make the test more demanding, increase maxmod and mtest.

   22 Aug 1991:  More statistics are now accumulated by LUTEST.
   29 Jun 1999:  dzero replaced by dload.
   29 Jun 1999:  Column replacement now done without reordering.
   30 Jun 1999:  Row and column delete swaps with last row and col.
   ------------------------------------------------------------------ */
  int    maxmod = MAXMOD;
  int    nrowb  = NROWB;
  int    ncolb  = NCOLB;

/*
  REAL B[NROWB*NCOLB + BLAS_BASE];       //  REAL B[NROWB,NCOLB];
  REAL C[MAXMOD*MAXMOD + BLAS_BASE];     //  REAL C[MAXMOD,MAXMOD];

  REAL D[MAXMOD + BLAS_BASE], x[MAXMOD + BLAS_BASE];
  REAL L[MAXMOD*MAXMOD + BLAS_BASE], U[MAXMOD*(MAXMOD+1)/2 + BLAS_BASE];
  REAL v[MAXMOD + BLAS_BASE];
  REAL w[NROWB + BLAS_BASE];
  REAL y[MAXMOD + BLAS_BASE], z[MAXMOD + BLAS_BASE];
*/

  int    seeds[3+BLAS_BASE];

  REAL   r1, r2, densty;
  int    m, n, mtest, verbose, resized;

  REAL   *B, *C, *D, *x, *L, *U, *v, *w, *y, *z;

/* Prepare BLAS */

  if(!load_BLAS(NULL))
    return;

/* Set model size and the number of tests to run */

  verbose =   0;
  mtest   =   1;
  densty  =   0.25;
  resized = FALSE;

  n = 1;
  while ( n < argc) {

	  if(strcmp(argv[n], "-h") == 0) {
	  printf("LUMOD v2.0 by Michael Saunders - Dense matrix C version by Kjell Eikland");
	  printf("Usage: LUMOD -h -v {-fd|-hd|-md|-dd|-ld|-td} <size> <count>");
      return;
	}
    else if(strcmp(argv[n], "-v") == 0)
      verbose = 1;
    else if(strcmp(argv[n], "-fd") == 0)
      densty = 1.00;
    else if(strcmp(argv[n], "-hd") == 0)
      densty = 0.75;
    else if(strcmp(argv[n], "-md") == 0)
      densty = 0.50;
    else if(strcmp(argv[n], "-dd") == 0)
      densty = 0.25;
    else if(strcmp(argv[n], "-ld") == 0)
      densty = 0.10;
    else if(strcmp(argv[n], "-td") == 0)
      densty = 0.03;
    else {
      if(resized) {
        mtest  = atoi(argv[n]);
      }
      else {
        maxmod  = atoi(argv[n]);
        nrowb   = 2*maxmod;
        ncolb   = 2*maxmod;
        resized = TRUE;
      }
    }
	n++;

  }


  B = (REAL *) calloc(nrowb*ncolb + BLAS_BASE, sizeof(REAL));
  C = (REAL *) calloc(maxmod*maxmod + BLAS_BASE, sizeof(REAL));
  D = (REAL *) calloc(maxmod + BLAS_BASE, sizeof(REAL));
  x = (REAL *) calloc(maxmod + BLAS_BASE, sizeof(REAL));
  L = (REAL *) calloc(maxmod*maxmod + BLAS_BASE, sizeof(REAL));
  U = (REAL *) calloc(maxmod*(maxmod+1)/2 + BLAS_BASE, sizeof(REAL));
  v = (REAL *) calloc(maxmod + BLAS_BASE, sizeof(REAL));
  w = (REAL *) calloc(nrowb + BLAS_BASE, sizeof(REAL));
  y = (REAL *) calloc(maxmod + BLAS_BASE, sizeof(REAL));
  z = (REAL *) calloc(maxmod + BLAS_BASE, sizeof(REAL));

/* Initialize the random number sequence. */

   r1       = -10.0;
   r2       =  10.0;
   randomseed(seeds);

/* Test square systems. */

   m      = maxmod;
   n      = maxmod;
   lutest( m, n, nrowb, ncolb, mtest,
           r1, r2, densty, seeds,
           B, C, D, L, U, v, w,
           x, y, z, verbose);


/* Free memory */

  free(B);
  free(C);
  free(D);
  free(x);
  free(L);
  free(U);
  free(v);
  free(w);
  free(y);
  free(z);

  unload_BLAS(NULL);

/*     End of main program for LUTEST */
}

