function [ x, flag, iter, rnorm, Arnorm, Anorm, Acond, xnorm, Axnorm...
         resvec,  Aresvec, xnormvec, pnormvec,  gamavec, alfavec, betavec, VV,...
         resvecdirect, relresvec, Aresvecdirect, relAresvec, pnormdirectvec,...
         tknormvec] =...
  minresSOL69( A, b, rtol, maxit, M, shift, maxxnorm, Acondlim, show, testsym, fignum, ...
               debug, disable, isbestiter, varargin )

% minresSOL69: Find a solution to symmetric Ax=b or min||Ax-b||
%        
% USAGE:
%          x = minresSOL69(A,b);
%   
%     or
%
%        [ x, flag, iter, relres, relAres, Anorm, Acond, xnorm] = ...
%        minresSOL69( A, b, rtol, maxit, M, shift, maxxnorm, Acondlim, show);
% 
%     or
% 
%        [ x, flag, iter, rnorm, Arnorm, Anorm, Acond, xnorm,...
%          resvec,  Aresvec, xnormvec, pnormvec,  gamavec, alfavec, betavec, VV,...
%          resvecdirect, relresvec, Aresvecdirect, relAresvec, pnormdirectvec] =...
%       minresSOL69( A, b, rtol, maxit, M, shift, maxxnorm, Acondlim, show, fignum, ...
%                debug, isbestiter, p1, p2,... )
%            passes parameters p1, p2,...to functions: AFUN(x, p1, p2,...) and
%            MFUN(x, p1, p2,...).    
%
% DESCRIPTION:
%     minresSOL69 attempts to find the minimum-length and minimum-residual-norm
%     solution x to the system of linear equations A*x = b or 
%     least squares problem min||Ax-b||.  The n-by-n coefficient matrix A 
%     must be symmetric (but need not be positive definite or invertible).  
%     The right-hand-side column vector b must have length n. A may be a 
%     function returning A*x.  A preconditioner M may be supplied as a
%     symmetric positive definite matrix or a function returning M\x.
%
% INPUTS:
%    A          n x n symmetric matrix, singular or invertible,  
%               real or complex, double or single. 
%               A may also be a function returning A*x.
%    b          n-vector, double or single, real or complex.                
%    rtol       Optional, real, specifies the tolerance of the method.
%               Default is 1e-6
%    maxit      Optional, positive integer, specifies the maximum number of
%               iterations. Default is 20
%    M          Optional, n x n symmetric positive definite preconditioner, 
%               real or complex, double or single. Effectively solve the
%               system 
%                  inv(sqrt(M)) * A * inv(sqrt(M)) * y = inv(sqrt(M)) * b 
%               for y and then x = inv(sqrt(M)) * y.  If M is [] or not provided,
%               then no preconditioner is applied.  M may also be a
%               function returning M\x.
%    shift      Optional, scalar, real or complex.  Default is 0.
%               Effectively solve the system (A - shift I) * x = b.
%    maxxnorm   real positive, maximum bound on NORM(x). Default is 1e14.
%    Acondlim   real positive, maximum bound on COND(A). Default is 1e15.
%    show       boolean, 0 to suppress outputs, 1 to show iterations.
%               Default is 0.
%    p1, p2,... Optional, inputs to A and M if they are functions
%
% OUTPUTS:
%    x       n-vector, estimated solution 
%    flag    integer, convergence flag
%           -1  beta2 = 0.  If M = I, b and x are eigenvectors.             
%            0 beta1 = 0.  The exact solution is  x = 0.                    
%            1 A solution to (poss. singular) Ax = b found, given rtol.     
%            2 Pseudoinverse solution for singular LS problem, given rtol.  
%            3 A solution to (poss. singular) Ax = b found, given eps.    
%            4 Pseudoinverse solution for singular LS problem, given eps.   
%            5 x has converged to an eigenvector.                         
%            6 xnorm has exceeded maxxnorm.                                
%            7 Acond has exceeded Acondlim.                                 
%            8 The iteration limit was reached.                             
%            9 It is a least squares problem but no converged solution yet.
%    iter    integer, iteration number at which x was computed: 0 <= iter <= maxit.
%    relres  real positive, the relative residual is defined as
%                 NORM(b-A*x)/(NORM(A) * NORM(x) + NORM(b)),
%            computed recurrently here.  If flag is 1 or 3,  relres <= TOL.
%    relAres real positive, the relative-NORM(Ar) := NORM(Ar) / NORM(A) ---
%            computed recurrently here. If flag is 2 or 4, relAres <= TOL. 
%    Anorm   real positive, estimate of matrix 2-norm of A.
%    Acond   real positive, estimate of condition number of A with
%            respect to 2-norm. 
%    xnorm   non-negative positive, recurrently computed NORM(x)
%    Axnorm  non-negative positive, recurrently computed NORM(A * x).
%
% EXAMPLE 1: 
%     n = 100; on = ones(n,1); A = spdiags([-2*on 4*on -2*on],-1:1,n,n);
%     b = sum(A,2); rtol = 1e-10; maxit = 50; M = spdiags(4*on,0,n,n);
%     x = minresSOL69(A, b, rtol, maxit, M);
%   
%     Use this matrix-vector product function
%        function y = afun(x,n)
%        y = 4 * x;
%        y(2:n) = y(2:n) - 2 * x(1:n-1);
%        y(1:n-1) = y(1:n-1) - 2 * x(2:n);
%     as input to minresSOL69
%        x1 = minresSOL69(@afun, b, rtol, maxit, M);
%
% EXAMPLE 2: A is Laplacian on a 50 by 05 grid, singular and indefinite.
%      n = 50; N = n^2; on=ones(n,1);   B = spdiags([on on on], -1:1, n, n);
%      A = sparse([],[],[],N,N,(3*n-2)^2);
%      for i=1:n 
%          A((i-1)*n+1:i*n,(i-1)*n+1:i*n) = B; 
%          if i*n+1 < n*n, A(i*n+1:(i+1)*n,(i-1)*n+1:i*n)=B; end; 
%          if (i-2)*n+1 > 0  A((i-2)*n+1:(i-1)*n,(i-1)*n+1:i*n)=B;  end; 
%      end
%      b = sum(A,2);   rtol = 1e-5;   maxxnorm = 1e2; 
%      shift = 0;   Acondlim = [];   show = 1;   M = [];  
%      x = minresSOL69( A, b, rtol, N, M, shift, maxxnorm, Acondlim, show);
%
% EXAMPLE 3: A is diagonal, singular and indefinite. 
%      h = 1;  a = -10; b = -a; n = 2*b/h + 1;   
%      A = spdiags((a:h:b)', 0, n, n);
%      b = ones(n,1);   rtol = 1e-6;   maxxnorm = 1e2; 
%      shift = 0;   Acondlim = [];   show = 1;   M = [];  
%      x = minresSOL69( A, b, rtol, N, M, shift, maxxnorm, Acondlim, show);
%
% SEE ALSO:
%    minresQLP, minresSOL, minresSOLs, minres, symmlq, lsqr, cgls
%    available as free software at 
%         http://www.stanford.edu/group/SOL/software.html
% 
% REFERENCES:
%    Sou-Cheng Choi's PhD Dissertation, Stanford University, 2006.
%    
% CURRENT / FUTURE RELEASES of minresSOL69:
%    Available at: http://www.stanford.edu/group/SOL/software.html
% 
% FUTURE WORK:
%    MM/DD/2006: ---
%
% MODIFICATION HISTORY:
% 02 Sep 2003: Date of Fortran 77 version, based on
%              C. C. Paige and M. A. Saunders (1975),
%              Solution of sparse indefinite systems of linear equations,
%              SIAM J. Numer. Anal. 12(4), pp. 617-629.
%
% 02 Sep 2003: ||Ar|| now estimated as Arnorm.
% 17 Oct 2003: f77 version converted to MATLAB.
% 28 Sep 2004: Stable symmetric Givens rotation:
%                 [ c  s ][a] = [d],
%                 [ s -c ][b]   [0]
%              where d = two norm of vector [a; b]
%              Implemented in SymGivens2.m.
% 27 Oct 2004: 1. if ~precon, more stable to use
%                    betan = norm(p);
%                 rather than
%                    betan = r2' * p;
%              2. guard against division by zero error in
%                    denom = 1/gama;
%                 by:  if ( gama <= epsa ), break main loop.
% 29 Oct 2004: 1. Changed relative residual norm from
%                  relrnorm  = rnorm / (Anorm*xnorm) % ||r||/(||A|| ||x||)
%              to
%                  relrnorm  = rnorm / (Anorm*xnorm + bnorm);
%                                          % ||r|| / (||A|| ||x|| + ||b||)
%              2. Also added the following two stopping conditions:
%                    if Arnorm <= epsa*beta1, flag = 2; end
%                    if betan  <= epsa      , flag = 1; end
% 19 Oct 2004: Exit main loop if betan == 0.  This is unlikely on
%              real problems but definitely happens on toy ones!
% 10 Nov 2004: 1. Grouped convergence tests together
%              2. Changed
%                     tnorm2 = tnorm2 + alfa^2 + beta^2 + betan^2;
%                 to the more numerical stable form calling NORM:
%                     Tnorm  = norm([Tnorm beta alfa betan]);
%                           % Frobenius norm of T_{k,k+1} estimates Anorm.
%                 Note that tnorm2 = Tnorm^2.
%                 So when iter == 1, now we have:
%                     Tnorm = norm([alfa betan]);
%                 It was miscalculated before in tnorm2:
%                     tnorm2 = alfa^2 + beta^2 + betan^2;
%                 (which may cause premature stop for some toy examples).
% 22 Nov 2004: Many end effects straightened out with Sou Cheng's 26
%              extreme 2x2 test cases, such as A = 1e+300*I (!).
% 24 Nov 2004: Compare Matlab's MINRES; we do not do
%              "Local reorthogonalization"
%  5 Nov 2004: Statistical test for symmetric matrices or operators is
%              implemented in IsOpSym6.m, in which we changed the test
%              statistics from
%                  z = | b'A'Ab - b'AAb |
%              to
%                  z = | b'(Ac) - c'(Ab) |
%  7 Jan 2005: 1. Added three lines:
%                 Dnorm = Dnorm + norm(d)^2;
%                 Acond  = Anorm * sqrt( Dnorm );
%                 str3 = sprintf( ' %8.1e %8.1e'      , Anorm, Acond );
%              2. Test run with new Matlab release 14.
% 31 Mar 2005: A must be a matrix, or a function handle.
% 31 Mar 2005: Added input argument fignum.
%              If fignum > 0, residual norms are plotted on figure(fignum).
% 06 Apr 2005: Rearranged the order of input parameters to the private
%              function minresxxxA
%                  p = minresxxxA( A, x, varargin )
%              to
%                  p = minresxxxA( A, x, varargin ).
%              Also applied to minresxxxM() and definition of definition
%              of A if it is passed in as a function handle.
%              The reason is we want the function handle to be compatible
%              with the requirement of Matlab's MINRES, which has the
%              following syntax and dictates that the iterates x has to be
%              the first input parameter of the function handle afun:
%                  minres(afun, b, rtol, maxit, mfun, shift, ...
%                        show, fignum, p1,p2,...)
%              passes parameters p1,p2,... to functions afun(x,p1,p2,...),
%              mfun(x,p1,p2,...).
% 07 Apr 2005: 1. Added the line
%                        x = (beta1/alfa) * v;
%                 to update x in case beta2 = 0.
%                 For example like A = diag([1e-15 1]), b = (0:1)', we
%                 have beta2 = 0, so
%                 min  || [alfa] y1 - [beta1] ||,
%                      || [ 0  ]    - [  0  ] ||
%                 y1 = beta1/alfa1 and x1 = V1 y1 = beta1/alfa1 v1.
%              2. To speed up graphics performance, when fignum > 0,
%                 preallocated memory by adding the line
%                      resvec = zeros(maxit,1);
%                 and plot resvec outside the main iterations, rather
%                 than in each iteration.
%              3. To speed up performance, commented out lines involving
%                 NNZ().
% 21 Apr 2005: 1. Made input and output order of MINRES similar to those
%                 of Matlab's MINRES:
%              Before:
%      [ x, flag, iter, rnorm, Arnorm, Anorm, Acond, xnorm ] = ...
%      minres(n, b, A, M, iw, rw, precon, shift, show, maxit, rtol, fignum)
%              After:
%      [ x, flag, iter, rnorm, Arnorm, Anorm, Acond, xnorm, resvec ] = ...
%      minres( A,  b,  rtol, maxit, M, shift, show, fignum, varargin )
%                  As a result, we also changed private functions, for
%                  example, from
%                       p = A( n, x, iw, rw )
%                  to
%                       p = A( x, varargin ).
%               2. Extended documentation with reference to the
%                  documentation of Matlab's MINRES
%               3. Rename function from minres<k>.m to minresSOL<k>.m
%               4. Renamed variables: Aname to A, Mname to M,
%                  ynorm to xnorm, itnlim to maxit, rnormk to resvec,
%                  iter to iter, istop to flag,
%               5. Checked on number of input arguments and when some
%                  inputs are missing, assign default values
% 28 Apr 2005: Symmetry check is no longer optional(!)
%              because failure to recognize asymmetry is disastrous!!
%              1. Checked nargin and input arguments, set default values of
%              inputs if not available from user
%              2. Added 2 output arguments Aresvec and xnormvec, which
%                 stores ||Ar_k|| and ||x_k|| repectively
%  2 Jun 2005:  Made "sections" of code called cells (a new feature
%                 in Matlab 7 marked by %% )
%  6 Jul 2005: 1. Removed variable qrnorm
% 15 Jul 2005: 1. Updated development history
% 29 Jul 2005: 1. Added recurrence relations for r_k and A r_k
% 04 Aug 2005: 1. Renamed 3 variables: d to d, w1 to d1, w2 to d2.
%             *2. Enabled MINRES to compute minimal-length solution
%                 for singular (compatible) linear system.
%              3. Added boolean input variable debug
% 18 Aug 2005: 1. Added input parameter maxxnorm to prevent exploding
%                 solution from singular problems. A good value for 
%                 maxxnorm is 1e5 if ||x_k|| < 1e5 is desired.
%                 xnorm seems to be a tight upper bound of xnorm from
%                 our numerical experiments.
% 27 Jan 2006: 1. Debugged preconditioned-MINRES.
% 24 Feb 2006: 1. Made minresSOL capable of doing single precision 
%                 calculations, meaning all intermediate matrices & vectors 
%                 are single if inputs A & b are single variables:
%                       x = minresSOL(single(A), single(b)) 
%              2. Updated Modification History
%              3. Formatted the whole file.
% 25 Apr 2006: 1. Combined the first and second iterations. 
%              2. Combined the third and main iterations.
%              3. Reduced vectors in memory.
%
% KNOWN BUGS:
%    MM/DD/2006: ---
%
% NOTES:
%    MM/DD/2006: ---
%
% AUTHORS: Terrya Sou-Cheng Choi, ICME, Stanford University
%          Michael Saunders, SOL, Stanford University
%
% CREATION DATE: 05/12/2006
%
% COPYRIGHT NOTICE:
%
%    This is Copyrighted Material. The software is COPYRIGHTED by the  
%    original authors.
% 
% COPYLEFT NOTICE:
%
%     Permission is granted to make and distribute verbatim copies of this 
%     file, provided that the entire file is copied **together** as a 
%     **unit**.
% 
%     The purpose of this permission notice is to allow you to make copies 
%     of the software and distribute them to others, for free or for a fee, 
%     subject to the constraint that you maintain the entire file here 
%     as a unit.  This enables people you give the software to be aware 
%     of its origins, to ask questions of us by e-mail, to request  
%     improvements, obtain later releases, and so forth.
% 
%     If you seek permission to copy and distribute translations of this 
%     software into another language, please e-mail a specific request to
%     saunders@stanford.edu and choi@sccm.stanford.edu.
% 
%     If you seek permission to excerpt a **part** of the software library 
%     for example to appear in a scientific publication, please e-mail a 
%     specific request to saunders@stanford.edu and choi@sccm.stanford.edu.
% 
% COMMENTS? 
%
%     E-mail saunders@stanford.edu and choi@sccm.stanford.edu
%
format short e
%%  Check inputs and set default values.
if nargin < 2
   error('Not enough input parameters')
end
if nargin < 3  || ~exist('rtol','var') || isempty(rtol)
   rtol   = 1e-6;
end
if nargin < 4  || ~exist('maxit','var') || isempty(maxit)
   maxit  = 20;
end
if nargin < 5  || ~exist('M','var') ||isempty(M)
  precon = false;
  M      = [];
else
  precon = true;
end
if nargin < 6  || ~exist('shift','var') ||isempty(shift)
  shift  = 0;
end
if nargin < 7  || ~exist('maxxnorm','var') || isempty(maxxnorm)
  maxxnorm = 1e14;
end
if nargin < 8  || ~exist('Acondlim','var') || isempty(Acondlim)
   Acondlim  =  0.1/eps;
end
if nargin < 9  || ~exist('show','var') || isempty(show)
   show   = true;
end
if nargin < 10  || ~exist('testsym','var') || isempty(testsym)
   testsym   = true;
end
if nargin < 11  || ~exist('fignum','var') || isempty(fignum)
   fignum = 0;
end
if nargin < 12 ||  ~exist('debug','var') ||isempty(debug)
   debug  = 0;
end
if nargin < 13 ||  ~exist('disable','var') ||isempty(disable)
   disable  = false;
end
if nargin < 14 ||  ~exist('isbestiter','var') ||isempty(isbestiter)
   isbestiter  = 0;
end

n     = length(b);
if isa(A,'float')
   classname = class(A);
else
   classname = 'double';
end

%%  Check outputs:   
%     resvec,  Aresvec, xnormvec, pnormvec,  gamavec, alfavec, betavec, VV,...
%     resvecdirect, relresvec, Aresvecdirect, relAresvec,
%     pnormdirectvec
if nargout > 9
   resvec         = zeros(maxit,1,classname);
else 
   resvec         = [];
end
if nargout > 10
   Aresvec   = zeros(maxit,1,classname);
else
   Aresvec   = [];
end
if nargout > 11
   xnormvec   = zeros(maxit,1,classname);
else
   xnormvec   = [];
end
if nargout > 12
   pnormvec   = zeros(maxit,1,classname);
else
   pnormvec   = [];
end
if nargout > 13
   gamavec = zeros(maxit,1);
else
   gamavec = [];
end
if nargout > 14
   alfavec = zeros(maxit,1);
else
   alfavec = [];
end
if nargout > 15
   betavec = zeros(maxit,1);
else
   betavec = [];
end
if (nargout > 16) || fignum
   VV   = zeros(n, maxit);
else
   VV = [];
end
if nargout > 17
   resvecdirect   = zeros(maxit,1,classname);
else
   resvecdirect   = [];
end
if nargout > 18
   relresvec   = zeros(maxit,1,classname);
else
   relresvec   = [];
end
if nargout > 19
   Aresvecdirect   = zeros(maxit,1,classname);
else
   Aresvecdirect   = [];
end
if nargout > 20
   relAresvec   = zeros(maxit,1,classname);
else
   relAresvec   = [];
end
if nargout > 21
   pnormdirectvec = zeros(maxit,1);
else
   pnormdirectvec = [];
end
if nargout > 22
   tknormvec = zeros(maxit,1);
else
   tknormvec = [];
end
if isbestiter
   allvecs = zeros(maxit,1);
else
   allvecs = [];
end

if fignum
   localOrthVtVmI  = zeros(maxit,1,classname);
   globalOrthVtVmI = zeros(maxit,1,classname);
end


%% Initalize

flag  = 0;
done  = false;
first = 'Enter minres-SOL.   ';
last  = 'Exit  minres-SOL.   ';
msg   =[' beta2 = 0.  If M = I, b and x are eigenvectors.              '   % -1
        ' beta1 = 0.  The exact solution is  x = 0.                    '   %  0
        ' A solution to (poss. singular) Ax = b found, given rtol.     '   %  1
        ' A least-squares solution was found, given rtol.              '   %  2
        ' A solution to (poss. singular) Ax = b found, given eps.      '   %  3
        ' A least-squares solution was found, given eps.               '   %  4
        ' x has converged to an eigenvector.                           '   %  5
        ' xnorm has exceeded maxxnorm.                                 '   %  6
        ' Acond has exceeded Acondlim.                                 '   %  7
        ' The iteration limit was reached.                             '   %  8
        ' A least-squares solution for singular LS problem, given eps. '   %  9
        ' A least-squares solution for singular LS problem, given rtol.'   %  10
        ' A null vector obtained, given rtol.                          ']; %  11

beta1 = norm(b);
if show
   disp( ' ' )
   disp( [ first 'Solution of symmetric Ax = b or min||Ax - b||.' ] )
   gprintf( sprintf( '\nn     = %3g  precon = %4g         shift   =  %10.2e     ||b||   =  %10.2e',...
                   n,           precon,              shift,               beta1) );
   gprintf( sprintf( '\nmaxit = %3g  rtol   = %11.2e  maxxnorm = %11.2e    Acondlim = %11.2e\n' ,...
                   maxit,       rtol ,           maxxnorm,            Acondlim) );
end

%---------------------------------------------------------------------
%% Decode A.
%---------------------------------------------------------------------
if isa(A,'float')         % A is an explicit matrix A, single or double
   [mA,nA] = size(A);
   if mA~=n || nA~=n
      error('A must be square & have the same dimension as length of b.')
   end
   if show,
      if issparse(A)
         fprintf('\nA is an explicit sparse matrix.')
      else
         fprintf('\nA is an explicit dense matrix.' )
      end
   end
elseif isa(A,'function_handle')
   disp(['The matrix A is defined by function_handle ' func2str(A) '.'])
else
   error('A must be a matrix or a function handle.')
end

%---------------------------------------------------------------------
%% Decode M.
%---------------------------------------------------------------------
if precon
   if isa(M,'float')       % M is an explicit matrix M.
      [mM,nM] = size(M);
      if mM~=n || nM~=n
         error('M must be square & have the same dimension as length of b.')
      end
      if show,
         if issparse(M)
            fprintf('\nM is an explicit sparse matrix.')
         else
            fprintf('\nM is an explicit dense matrix.' )
         end
      end
   elseif isa(M,'function_handle')
      disp(['The matrix M is defined by function_handle ' func2str(M) '.'])
   else
      error('M must be a matrix or a function handle.')
   end
end % if precon


%---------------------------------------------------------------------
%% Check if using single precision for variables and computations.
%---------------------------------------------------------------------
if isa(A, 'single') && isa(b, 'single')
   if precon && isa(M,'double'),
      M = single(M);
   end
   fprintf('\nComputations are done in single precision.')
end

% ------------------------------------------------------------------
%% Test A and M for symmetry.
% IsOpSym6 uses a statistical test.
% ------------------------------------------------------------------
if testsym
   [issym, Anorm] = IsOpSym6(A, n, varargin);   % See if A is symmetric.
   if ~issym
      if show, disp(' '); end
      error('The matrix or operator "A" appears to be unsymmetric');
   end

   if precon
      [issym, Mnorm] = IsOpSym6(M, n, varargin);   % See if M is symmetric.
      if ~issym
         if show, disp(' '); end
         error('The preconditioner "M" appears to be unsymmetric');
      end
   end
end
%------------------------------------------------------------------
%% Some people accidentally input a sparse b(!).
%------------------------------------------------------------------
if issparse(b)
   if show, disp(' '); end
   warning('b is a sparse vector.  MINRES will make it dense');
   b = full(b);
end

%------------------------------------------------------------------
%% Set up p and v for the first Lanczos vector v1.
% p  =  beta1 P' v1,  where  P = C**(-1).
% v is really P' v1.
%------------------------------------------------------------------
r3    = b;
r2    = b;
r1    = b;

if precon
   r3 = minresxxxM( M, b, varargin );
   if length(r3) ~= n
      error('Mfun must return a vector of the same length as b')
   end
   beta1 = r3'*b;
   %  Test for an indefinite preconditioner.
   %  If b = 0 exactly, stop with x = 0.
   if beta1 < 0
      error('The preconditioner "M" appears to be indefinite');
   else
      beta1  = sqrt( beta1 );
   end
end

if beta1==0  % b = 0, so x = 0
   flag = 0;  show = true;  done = true;
end

%------------------------------------------------------------------
%% Initialize other quantities.
% Note that Anorm has been initialized by IsOpSym6.
% ------------------------------------------------------------------

iter     = 0;       beta   = 0;      phi      = beta1;  
Acond    = 1;       betan  = beta1;  bnorm    = beta1;    
cs       = -1;      dbarn  = 0;      eplnn   = 0;          
rnorm    = betan;   sn     = 0;      Tnorm    = 0;        rnorml   = beta1;
xnorm    = 0;       Dnorm  = 0;      xnorm = 0;        xrArnormsl = 0;
prnt2    = true;    gama  = 0;       lastiter = 0;        pnorm      = 0;
bestiter = 0;       gamal = 0;       Axnorm = 0;
relrnorm = rnorm/beta1;              Anorm = 0;
BIG = 1000;         lines = 1;       headlines = 30 * lines;
if isbestiter
   xrArnormsbest = BIG * Anorm * rnorm^2;
end

dl      = minresxxxA( A, b, varargin );  % dl temporarily used for testing
if length(dl) ~= n
   error('Afun must return a vector of the same length as b')
end
Arnorm = norm(dl);

x        = zeros(n,1,classname);
d       = zeros(n,1,classname);
dl      = zeros(n,1,classname);

if fignum  
   mfig = 2;          nfig = 4;          figure(fignum);     clf;
end
if fignum || (nargout > 8)
   rnormdirect = beta1;   xnorm = 0;       pnormdirect = 0;
   Arnormdirectl = 0;  Arnormdirect = Arnorm;    Arnormbest = Arnorm; 
end

if show || debug
   disp(' ')
   disp(' ')
   head1 = '  iter  bestiter  x(1)   xnorm     rnorm     Arnorm   ';
   head2 = 'Compatible   LS    norm(A)  cond(A)  beta'; 
   disp([head1 head2])  
end

%---------------------------------------------------------------------
%% Main iteration loop.
% --------------------------------------------------------------------
while ~done 
    iter = iter + 1;
%-----------------------------------------------------------------
%% Obtain quantities for the next Lanczos vector vk+1, k = 1, 2,...
% The general iteration is similar to the case k = 1 with v0 = 0:
%
%   p1      = Operator * v1  -  beta1 * v0,
%   alpha1  = v1'p1,
%   q2      = p2  -  alpha1 * v1,
%   beta2^2 = q2'q2,
%   v2      = (1/beta2) q2.
%
% Again, p = betak P vk,  where  P = C**(-1).
% .... more description needed.
%-----------------------------------------------------------------
    betal    = beta;
    beta     = betan;  
    v        = r3 * (1/beta); 
    r3       = minresxxxA( A,  v, varargin );% p_k
    if shift ~= 0
       r3    = r3 - shift*v;
    end
    if fignum || ~isempty(pnormdirectvec) 
       pnormdirectl = pnormdirect;
       if ~precon
          pnormdirect  = norm(r3);
       else
          pnormdirect  = sqrt( r3' * minresxxxM( M, r3, varargin ));
       end
    end
    if iter  > 1,
       r3    = r3 - (beta/betal) * r1;
    end
    alfa     = real( r3'*v );                         % alpha_k
    r3       = r3 - (alfa/beta ) * r2; % v_{k+1}
    r1       = r2;                            % v_{k-1}
    r2       = r3; 

   if ~precon,
      betan  = norm(r3);                      % beta_{k+1}
    else
      r3     = minresxxxM( M, r2, varargin );
      betan  = r2' * r3;
      if betan < 0
         error('The preconditioner "M" appears to be indefinite');
      end
      if betan > 0,
         betan = sqrt( betan );     
      end
    end
    pnorml  = pnorm;
    if iter == 1        
       pnorm  = norm([alfa, betan]);
    else
       pnorm  = norm([beta, alfa, betan]);
    end
    if debug
       fprintf('\n  v_%d     = ', iter );  fprintf('%13.3e ', v(1:min(n,5)) );
       fprintf('\n  alpha_%d = %13.3e, beta_%d = %13.3e, beta_%d = %13.3e ',...
                    iter, alfa, iter, beta, iter+1, betan );
    end
%-----------------------------------------------------------------
%% Apply previous rotation Qk-1 to get
%   [dlta_k epln_{k+1}] = [cs  sn][dbar_k    0      ]
%   [gbar_k  dbar_{k+1} ]   [sn -cs][alfa_k beta_{k+1}].
%-----------------------------------------------------------------
    dbar   = dbarn;
    epln  = eplnn;

    dlta  = cs*dbar + sn*alfa;      % dlta1 = 0         dltak
    gbar  = sn*dbar - cs*alfa;      % gbar 1 = alfa1     gbar k

    eplnn =          sn*betan;      % epln2 = 0        eplnk+1
    dbarn =        - cs*betan;      % dbarn 2 = beta2   dbarn k+1

    if debug
       fprintf('\n  c_%d     = %13.3e, s_%d    = %13.3e ', ...
                    iter-1, cs, iter-1, sn );
       fprintf('\n  dlta_%d = %13.3e, gbar_%d = %13.3e ', ...
                    iter, dlta, iter, gbar );
       fprintf('\n  epln_%d = %13.3e, dbar_%d = %13.3e ', ...
                    iter+1, eplnn, iter+1, dbarn );
    end

%% Compute the current plane rotation Qk
    gamal2 = gamal;
    gamal  = gama;
    [cs, sn, gama] = SymGivens2(gbar, betan);
    tau     = cs*phi;               % phik
    phi     = sn*phi;               % phibar_{k+1}
     Axnorm = norm([Axnorm  tau]); % norm([tau_1,..., tau_k])
    if debug
       fprintf('\n  c_%d     = %13.3e, s_%d    = %13.3e ', iter, cs, iter, sn );
       fprintf('\n  tau_%d   = %13.3e, phi_%d  = %13.3e ', iter, tau, iter, phi);
       fprintf('\n  gama_%d = %13.3e               ', iter, gama );
    end
    
%% Update d

    dl2       = dl;
    dl        = d;
    if gama  ~= 0,
       d      = (v - epln*dl2 - dlta*dl) * (1/gama);
       d_norm = norm(d);
     else
       d_norm = Inf;
    end    

%% Update x except if it will become too big
   x1last      = x(1);
   xnorml      = xnorm;
   dl2         = x;   % temporarily using dl2 to store x_{k-1}
   x           = x + tau*d;
   xnorm       = norm(x);
   if xnorm >= maxxnorm %max(d_norm, xnorm) >= maxxnorm
      x        = dl2;  
      flag     = 6;
      lastiter = 1;
   end
   
   if debug
      fprintf('\n  w_%d     = ', iter ); fprintf('%13.3e ', d(1:min(n,5)) );
      fprintf('\n  x_%d     = ', iter ); fprintf('%13.3e ', x(1:min(n,5)) );
      fprintf('\n  xnorm = %13.3e ', xnorm);
   end

%% Estimate various norms
   %xnorml      = xnorm;
   rnorml      = rnorm;                         % ||r_{k-1}||
   Anorml      = Anorm;
   Acondl      = Acond;
   relrnorml   = relrnorm;
   
   if flag ~= 6;
      Dnorm    = norm([Dnorm, d_norm]);
       
      xnorm    = norm(x);                       % rather expensive
      
      rnorm    = phi;                           % ||r_k||
      relrnorm = rnorm  / (Anorm*xnorm + bnorm); % ||r||/(||A||||x||+||b||)
      
      if iter  == 1
         Tnorm = norm([alfa, betan]);
      else
         Tnorm = norm([Tnorm; beta; alfa; betan]); % Frobenius norm of T_{k,k+1}
      end                                      % used as estimate of Anorm.
      %Anorm    = Tnorm;
      Anorm = max([Anorm, pnorm]);
      Acond    = Anorm * Dnorm;
   end 

    if fignum && (iter > 1)
      Arnorml2    = Arnorml;
      relArnorml2 = relArnorml;
    end
    rootl       = norm([gbar; dbarn]);
    Arnorml     = rnorml*rootl;             % ||A r_{k-1} ||
    relArnorml  = rootl  /  Anorm;          % ||Ar|| / (||A|| ||r||)     
    %relArnorml = Arnorml / Anorm;          % ||Ar|| / ||A|| 
    
    if isbestiter,
       isbestiter  = xnorml * rnorml * Arnorml;
       if xrArnormsbest > xrArnormsl,
          bestiter      = iter - 1;
          xbest         = dl2;
          xnormbest     = xnorml; 
          rnormbest     = rnorml;
          Arnormbest    = Arnorml;
          relrnormbest  = relrnorml;
          relArnormbest = relArnorml;
          if iter > 1, xrArnormsbest = xrArnormsl; end    
       end
    end
    
    if fignum || (nargout > 8) 
      rnormdirectl  = rnormdirect;
      rdirect       = b - minresxxxA( A, x, varargin ) + shift*x;
      if ~precon
         rnormdirect   = norm(rdirect);
      else
         Mrdirect      = minresxxxM( M, rdirect, varargin );
         rnormdirect   = sqrt( rdirect' * Mrdirect);
      end
      Arnormdirectl2 = Arnormdirectl;
      Arnormdirectl  = Arnormdirect;
      if ~precon
          Arnormdirect  = norm(minresxxxA( A, rdirect,varargin ) - shift* rdirect);
      else
          Mrdirect      = minresxxxA( A, Mrdirect,varargin ) - shift * Mrdirect;   
          Arnormdirect  = sqrt( Mrdirect' *  minresxxxM( M, Mrdirect, varargin ) );
      end
      if ~isempty(resvec)
         resvec(iter) = rnorm;
      end
      if ~isempty(relresvec)
         relresvec(iter) = relrnorm;
      end 
      if ~isempty(resvecdirect)
         resvecdirect(iter) = rnormdirect;
      end
      if (iter > 1) 
         if ~isempty(Aresvec) 
            Aresvec(iter-1) = Arnorml; 
         end

         if ~isempty(relAresvec) 
            relAresvec(iter-1) = relArnorml;
         end
         if ~isempty(gamavec) 
            gamavec(iter-1)   = gamal;
         end
      end
      if ~isempty(Aresvecdirect) 
         Aresvecdirect(iter) = Arnormdirect;
      end
      if ~isempty(xnormvec)
         xnormvec(iter) = xnorm;
      end
      if ~isempty(pnormvec)
         pnormvec(iter) = pnorm;
      end
      if ~isempty(pnormdirectvec)
         pnormdirectvec(iter) = pnormdirect;
      end
      if ~isempty(tknormvec)
         tknormvec(iter) = Axnorm;
      end
      if isbestiter
         allvecs(iter)     = xnorm * relrnorm * relArnorml;
      end  
      if ~isempty(gamavec)
         gamavec(iter) = gama;
      end
      if ~isempty(alfavec) 
         alfavec(iter)        = alfa;
      end
      if ~isempty(betavec) 
         betavec(iter)        = betan;
      end
      if ~isempty(VV) 
         VV(:,iter)           = v;
      end
   end
   
    if debug
       fprintf('\n  Tnorm_%d  = ', iter );  fprintf('%13.3e ', Tnorm );
       fprintf('\n  rnorm_%d  = ', iter );  fprintf('%13.3e ', rnorm );
       fprintf('\n  Ar_norm%d = ', iter-1); fprintf('%13.3e ', Arnorml);
       fprintf('\n  x_norm%d  = ', iter );  fprintf('%13.3e ', xnorm );
       fprintf('\n  Acond%d   = ', iter );  fprintf('%13.3e ', Acond );
       fprintf('\n\n');
    end

%---------------------------------------------------------------
%% See if any of the stopping criteria are satisfied.
% In rare cases, flag is already -1 from above (Abar = const*I).
%---------------------------------------------------------------
   % epsa   = Anorm * eps;            % Used below to test betan and ||A r||
    epsx   = Anorm * xnorm * eps;
    epsr   = Anorm * xnorm * rtol;  
    
    % Test for singular Hk (hence singular A)
    % or x is already an LS solution (so again A must be singular).

    if flag==0 || flag == 6
       t1 = 1 + relrnorm;             % These tests work if rtol < eps
       t2 = 1 + relArnorml;
       if rtol >= eps
          if     t1         <= 1       , flag = 3;  % Accurate Ax=b solution
          elseif t2         <= 1       , flag = 4;  % Accurate LS solution
          elseif relrnorm   <= rtol    , flag = 1;  % Good enough Ax=b solution
          elseif relArnorml <= rtol    , flag = 10; % Good enough LS solution
          elseif epsx       >= beta1   , flag = 5;  % x is an eigenvector
          elseif xnorm      >= maxxnorm, flag = 6;  % xnorm exceeded its limit
          elseif Acond      >= Acondlim, flag = 7;  % Huge Acond (but maybe ok?)
          elseif iter       >= maxit   , flag = 8;  % Too many itns
          end
        else 
          if     relrnorm   <= rtol    , flag = 1;   % Good enough Ax=b solution
          elseif relArnorml <= rtol    , flag = 10;  % Good enough LS solution
          elseif epsx       >= beta1   , flag = 5;   % x is an eigenvector
          elseif xnorm      >= maxxnorm, flag = 6;   % xnorm exceeded its limit
          elseif Acond      >= Acondlim, flag = 7;   % Huge Acond (but maybe ok?)
          elseif iter       >= maxit   , flag = 8;   % Too many itns
          end
       end
    end
    if disable 
       if (iter >= maxit) flag == 8; done = 1;
       else flag = 0;  done = 0; end
       
       %Axnorm = norm(A*x);
       if Axnorm < rtol * Anorm * xnorm
          flag = 11;   done = 1;
          lastiter = 0;
       end
    end
    if flag ~= 0 % flag may have been changed above
       done      = 1;
       if (flag == 1) || (flag == 3) || (flag == 8) || (flag == 11) % Linear systems
          r1        = b - minresxxxA( A, x , varargin ) + shift*x; %use r1 to temporarily store residual
          Arnorm    = norm(minresxxxA( A, r1, varargin ) - shift * r1); % avoid computing 1 more Lanczos iteration to get root
          relArnorm = Arnorm / (Anorm * rnorm);                % ||Ar|| / ||A||
       end  
       if (flag == 4) || (flag == 6) ||  (flag == 10) || (flag == 7)
          lastiter = 1;
       end
       if lastiter == 1
          iter      = iter - 1; 
          prnt2     = false;
          rnorm     = rnorml;
          relrnorm  = relrnorml;
          Arnorm    = Arnorml;
          relArnorm = relArnorml;
          pnorm     = pnorml; 
          gama      = gamal;
          gamal     = gamal2;
          if fignum || (nargout > 8) 
             Arnormdirect = Arnormdirectl;
             Arnormdirectl = Arnormdirectl2;
          end
       end 
       if flag == 9
          prnt2 = true;
       end
       if ~isempty(Aresvec) & iter > 1 
          Aresvec(iter) = Arnorm;
       end
       if ~isempty(relAresvec) & iter > 1
          relAresvec(iter) = relArnorm;
       end
    end

%% See if it is time to print something.

    prnt       = false;
    if n      <= 40       , prnt = true; end
    if iter   <= 10       , prnt = true; end
    if iter   >= maxit-10 , prnt = true; end
%     if rnorm  <= 10*epsx  , prnt = true; end
%     if rnorm  <= 10*epsr  , prnt = true; end
%     if Acond  <= 1e-2/eps , prnt = true; end
    
     
    if mod(iter-1,lines)==0  , prnt = true; end
    if flag   ~= 0        , prnt = true; end

    if debug,
       disp([head1 head2]);
    end
    
    if show && prnt && prnt2
       str1 = sprintf( '\n%6g %6g %10.2e %10.2e %10.2e %10.2e %10.3e', ...
                        iter-1, bestiter, x1last, xnorml, rnorml,...
                        Arnorml, relrnorml );
       str2 = sprintf( ' %10.3e'           , relArnorml);
       str3 = sprintf( ' %8.1e %8.1e %8.1e', Anorml, Acondl, betal);
       gprintf([str1 str2 str3 ]);
    if iter == 101     
       lines = 10;     headlines = 30 * lines;
    elseif iter == 1001     
       lines = 100;     headlines = 30 * lines;   end
    if mod(iter, headlines) == 0     
       fprintf('\n\n%s\n', [head1 head2]);     
    end
    end % show && prnt
    
%% plot figures
   if fignum && prnt && (iter > 1) && ~lastiter
      xind  = iter-1:iter;
      xindl = xind -1;
      subplot(mfig,nfig,1);
      semilogy(xind, max([rnorml rnorm], realmin), '-r.');   hold on;  
      semilogy(xind, max([relrnorml relrnorm], realmin), '-.k'); 
    
      if iter > 2
         subplot(mfig,nfig,2);
         semilogy(xindl, max([Arnorml2 Arnorml], realmin), '-r.'); hold on;
         semilogy(xindl, max([relArnorml2 relArnorml], realmin), '-.k');
      end
      
      subplot(mfig,nfig,3);
      semilogy(xind, max([xnorml xnorm],realmin), '-r.');   hold on;
       
      subplot(mfig,nfig,4);
      plot(xind, [pnorml pnorm] , '-r.');  hold on;
       
      subplot(mfig,nfig,5);
      semilogy(xindl, [gamal2 gamal], '-r.');   hold on;
      semilogy(xind, [Anorml Anorm], '--b.');
      semilogy(xind, [Acondl Acond], '-k.');

      subplot(mfig,nfig,6);
      loglog([xnorml xnorm], [rnorml rnorm], '-r.'); hold on;
   end
end % main loop

xnorm   = norm(x);  % So far, xnorm is an estimate of sqrt(x'Mx)

if isbestiter,
   if xrArnormsbest >= xnorm*rnorm*Arnorm,
      bestiter       = iter;
      xbest          = x;  
      xnormbest      = xnorm; 
      rnormbest      = rnorm;
      Arnormbest     = Arnorm;
      relrnormbest   = relrnorm;
      relArnormbest  = relArnorm;
   end
end

%% print final iteration
if show && prnt 
   str1 = sprintf( '\n%6g %6g %10.2e %10.2e %10.2e %10.2e %10.3e', ...
                   iter, bestiter, x(1), xnorm, rnorm, Arnorm, relrnorm);
   str2 = sprintf( ' %10.3e'           , relArnorm);
   str3 = sprintf( ' %8.1e %8.1e %8.1e ', Anorm, Acond, beta);
   str4 = sprintf( '\n%98.1e               ',                 betan);
   gprintf([str1 str2 str3 str4]);
end % show && prnt

%% Display final status.

if debug
   fprintf('\n\n');
end
if show
   v = b - minresxxxA( A,x,varargin ) + shift*x;
   disp( ' ' )
   gprintf( [last sprintf( ' flag    =  %2g               iter =%4g\n', ...
                          flag,                 iter   )] );
   gprintf( [last sprintf( ' Anorm   =  %12.4e     Acond =  %12.4e\n' , ...
                          Anorm,                Acond  )] );
   gprintf( [last sprintf( ' rnorm   =  %12.4e     rnorm direct  =  %12.4e\n' ,...
                          rnorm,                norm(v)  )] );
   gprintf( [last sprintf( ' Arnorm  =  %12.4e     Arnorm direct =  %12.4e\n',...
                          Arnorm,              norm(minresxxxA( A,v,varargin )-shift*v)  )] );
   gprintf( [last sprintf( ' xnorm   =  %12.4e     xnorm direct  =  %12.4e\n' ,...
                          xnorm,               norm(x) )] );
   gprintf( [last msg(flag+2,:)]);
   disp( ' ' )
end

%% Plot residual norm, solution norm and norm of A r_k

if fignum
   VV = VV(:,1:iter);  %1's on diagonal; can have negative off-diagonal entries
   VtVmI = abs(VV' * VV - speye(iter,iter)); 
   localOrthVtVmI(1)  = eps; % v_1 orthogonal to v_0=0
   globalOrthVtVmI(1) = eps;
   for i = 3:iter  % v_i v_i-1 v_i-2
       if i > 2
          localOrthVtVmI(i)  = norm(VtVmI(i-2:i,i-2:i),'inf');
       else
          localOrthVtVmI(i)  = norm(VtVmI(1:2,1:2),'inf');
       end
       globalOrthVtVmI(i) = norm([ globalOrthVtVmI(i-1); VtVmI(1:i,i)],'inf');% norm( VtVmI(1:i,1:i),'inf');
   end
   localOrthVtVmI  = localOrthVtVmI(1:iter-1);
   globalOrthVtVmI = globalOrthVtVmI(1:iter-1);
end

if  ~isempty(resvec) 
   resvec      = resvec(1:iter); 
end
if ~isempty(resvecdirect) 
   resvecdirect= resvecdirect(1:iter);
end
if ~isempty(relresvec) 
   relresvec   = relresvec(1:iter);
end
if ~isempty(Aresvec) 
   Aresvec     = Aresvec(1:iter);
end
if ~isempty(relAresvec) 
   relAresvec  = relAresvec(1:iter);
end
if ~isempty(Aresvecdirect)
   Aresvecdirect= Aresvecdirect(1:iter);
end
if ~isempty(xnormvec)
   xnormvec    = xnormvec(1:iter);
end
if isbestiter
   allvecs     = allvecs(1:iter);
end
if ~isempty(gamavec)   
   gamavec     = gamavec(1:iter);
end
if ~isempty(alfavec)    
   alfavec     = alfavec(1:iter);
end
if ~isempty(betavec)  
   betavec     = betavec(1:iter);
end
if ~isempty(pnormvec) 
   pnormvec    = pnormvec(1:iter);
end
if ~isempty(pnormdirectvec) 
   pnormdirectvec  = pnormdirectvec(1:iter);
end
if ~isempty(tknormvec)
   tknormvec    = tknormvec(1:iter);
end
if fignum && (iter > 1)
   h = gcf;
   set(h, 'Position', [275   300   860   645]);
   options={'Interpreter','latex'}; 
   % plot last iteration, legends and axis labels
   h = subplot(mfig,nfig,1);
   semilogy(iter, max([rnorml rnorm], realmin), '-r.');   
   semilogy(iter, max([relrnorml relrnorm], realmin), '-k.')
   hl = legend('$\phi_k$', '$\phi_k/(\|A\|_2^{(k)} \chi_k +\|b\|_2)$', 'Location', 'SouthOutside' ); 
   set(hl, 'Interpreter','latex');
   title('MINRES-SOL69');
   xlabel('Iteration $k$', options{:});
   semilogy([1 iter], [rtol rtol], '-g'); 
   semilogy(iter, max(rnormdirect, realmin), 'rs'); 
   semilogy(iter, max(rnormdirect/(Anorm*xnorm+beta1), realmin), 'ks')
   axis tight; 
   axis square;
   
   h = subplot(mfig,nfig,2);
   semilogy(iter-1:iter, max([Arnorml Arnorm], realmin), '-r.'); hold on;
   semilogy(iter-1:iter, max([relArnorml relArnorm], realmin), '-.k');
   hl =  legend('$ \psi_k $','$ \psi_k / (\|A\|_2^{(k)} \phi_k) $', 'Location', 'SouthOutside'); 
   set(hl, 'Interpreter', 'latex');
   xlabel('Iteration $k$', options{:});  
   semilogy([1 iter], [rtol rtol], '-g');  
   semilogy(iter, max(Arnormdirect, realmin), 'rs'); 
   semilogy(iter, max(Arnormdirect/(Anorm*rnormdirect), realmin), 'ks')
   axis tight;
   axis square;
   set(h, 'Position', [3.4257e-001  5.8384e-001  1.5327e-001  3.4116e-001]);
 
   h = subplot(mfig,nfig,3);
   semilogy(iter, max(xnorm,realmin), 'r.');   
   title('$\|x_k\|_2$', options{:});  
   xlabel('Iteration $k$', options{:});  
   axis tight
   axis square;
   set(h, 'Position', [5.5566e-001  5.8384e-001  1.5327e-001  3.4116e-001]);
     
   h = subplot(mfig,nfig,4);
   plot(iter, pnorm  , 'r.');   
   plot(iter, pnormdirect, 'rs'); 
   title('$\|T_k e_k\|_2 \simeq \|p_k\|_2$', options{:});
   xlabel('Iteration $k$', options{:});
   axis tight;
   axis square;
   set(h, 'Position', [7.4835e-001  5.8384e-001  1.5327e-001  3.4116e-001]);
   
   h = subplot(mfig,nfig,5);
   plot(iter-1:iter, abs([gamal gama]), '-r');   hold on;
   semilogy(iter, Anorml, 'b.');
   semilogy(iter, Acondl, 'k.');
   hl = legend('$\gamma_k$','$\|A\|_2$','$\kappa_2(A)$', 'Location', 'SouthOutside');
   set(hl, 'Interpreter','latex');
   xlabel('Iteration $k$', options{:});  
   axis tight;
   axis square; 
   set(h, 'Position', [1.3000e-001  1.5521e-001   1.5327e-001  3.4116e-001]);
   
   h = subplot(mfig,nfig,6);
   loglog(xnorm, rnorm, 'r.');  
   loglog(xnorm, rnormdirect, 'rs'); 
   ylabel('$\phi_k$', options{:});
   xlabel('$\|x_k\|_2$', options{:});  
   axis tight;
   axis square;
   set(h, 'Position', [3.7745e-001  1.5521e-001   1.5327e-001  3.4116e-001]);
   
   if iter > 2
      h = subplot(mfig,nfig,7);
      semilogy(1:iter-1,  max(localOrthVtVmI, eps), '-r');  hold on; 
      semilogy(1:iter-1, max(globalOrthVtVmI, eps), '--b');   
      title('$\| V_k^T V_k - I \|_\infty$', options{:});  
      legend('local','global','Location', 'SouthOutside');
      xlabel('Iteration $k$', options{:});    
      axis tight
      axis square
      set(h, 'Position', [5.4869e-001  1.5521e-001   1.5327e-001  3.4116e-001]);
   
      h = subplot(mfig,nfig,8);
      %imagesc(VtVmI)
      %colorbar(10.^[-15:1])
      cspy3(VtVmI(1:iter-1, 1:iter-1), 0, 5);
      title('CSPY( $| V_k^T V_k - I |$ )', options{:});
      xlabel('$k$', options{:});
      ylabel('$k$', options{:});   
      axis tight;
      axis square;
      set(h, 'Position', [7.4835e-001  1.5521e-001   1.5327e-001  3.4116e-001]);
   end
end

clear minresxxxM

%-----------------------------------------------------------------------
%% End function minres.m
%-----------------------------------------------------------------------
%% private functions

function p = minresxxxA( A,x,varargin )

%        p = minresxxxA( A,x,varargin )
% computes p = Ax for a matrix or operator A defined by input parameter A.

%-----------------------------------------------------------------------
% 17 Oct 2003: Default A*x function for minres.m.
%              User parameters A, varargin are passed thru to here.
%-----------------------------------------------------------------------

  if isa(A,'function_handle')
     p = A( x,varargin );
  else
     p = A*x;
  end

%-----------------------------------------------------------------------
% End private function minresxxxA
%-----------------------------------------------------------------------

function p = minresxxxM( M,x,varargin )

%        p = minresxxxM( M,x,varargin )
% solves My = x for a matrix or operator M defined by input parameter M.

%-----------------------------------------------------------------------
% 17 Oct 2003: Default M\x for minres.m.
%              User parameters M, varargin are passed thru to here.
% Precondition: M symmetric positive definite.
% Note: CHOL works with sparse matrices.
%-----------------------------------------------------------------------
  persistent R 
  if isa(M,'function_handle')
     p = M( x,varargin );
  else
     %p = M\x; % calls CHOL in each iteration!!!
     %The following calls CHOL only once throughout all iterations
     if ~exist('R','var') || isempty(R)
        % clear minresxxxM before existing MINRES-B.
        R = chol(M); 
     end 
     p = R' \ x;
     p = R  \ p;
  end

%-----------------------------------------------------------------------
% End private function minresxxxM
%-----------------------------------------------------------------------

function str = gprintf(str);
%-----------------------------------------------------------------------
%    format short e; a=.001234;
%    fprintf(regexprep(sprintf('%8.2E',a),'E-0','E-')) 
% gives 1.23E-03
%-----------------------------------------------------------------------
str = regexprep(str,'+0','+');
str = regexprep(str,'-0','-');
fprintf(str);
%-----------------------------------------------------------------------
% End private function gprintf
%-----------------------------------------------------------------------