function [ x, flag, iter, rnorm, Arnorm, Anorm, Acond, xnorm, Axnorm,...
           resvec,  Aresvec, xnormvec, pnormvec,  gamavec, alfavec, betavec, veplnvec, etavec, VV,...
           resvecdirect, relresvec, Aresvecdirect, relAresvec, xnormdirectvec, pnormdirectvec, tknormvec] = ...
  minresQLP49( A, b, rtol, maxit, M, shift, maxxnorm, Acondlim, show, testsym, fignum, ...
               debug, disable, isbestiter, varargin )

% minresQLP49: Find the pseudoinverse solution to symmetric Ax=b or min||Ax-b||
%        
% USAGE:
%          x = minresQLP49(A,b);
%   
%     or
%
%        [ x, flag, iter, relres, relAres, Anorm, Acond, xnorm] = ...
%        minresQLP49( A, b, rtol, maxit, M, shift, maxxnorm, Acondlim, show);
% 
%     or
% 
%        [ x, flag, iter, relres, relAres, Anorm, Acond, xnorm] = ...
%        minresQLP49( AFUN, b, rtol, maxit, MFUN, shift, maxxnorm, Acondlim, show, p1, p2,...);
%            passes parameters p1, p2,...to functions: AFUN(x, p1, p2,...) and
%            MFUN(x, p1, p2,...).    
%
% DESCRIPTION:
%     minresQLP49 attempts to find the minimum-length and minimum-residual-norm
%     solution x to the system of linear equations A*x = b or 
%     least squares problem min||Ax-b||.  The n-by-n coefficient matrix A 
%     must be symmetric (but need not be positive definite or invertible).  
%     The right-hand-side column vector b must have length n. A may be a 
%     function returning A*x.  A preconditioner M may be supplied as a
%     symmetric positive definite matrix or a function returning M\x.
%
% INPUTS:
%    A          n x n symmetric matrix, singular or invertible,  
%               real or complex, double or single. 
%               A may also be a function returning A*x.
%    b          n-vector, double or single, real or complex.                
%    rtol       Optional, real, specifies the tolerance of the method.
%               Default is 1e-6
%    maxit      Optional, positive integer, specifies the maximum number of
%               iterations. Default is 20
%    M          Optional, n x n symmetric positive definite preconditioner, 
%               real or complex, double or single. Effectively solve the system 
%                  inv(sqrt(M)) * A * inv(sqrt(M)) * y = inv(sqrt(M)) * b 
%               for y and then x = inv(sqrt(M)) * y.  If M is [] or not provided,
%               then no preconditioner is applied.  M may also be a
%               function returning M\x.
%    shift      Optional, scalar, real or complex.  Default is 0.
%               Effectively solve the system (A - shift I) * x = b.
%    maxxnorm   real positive, maximum bound on NORM(x). Default is 1e14.
%    Acondlim   real positive, maximum bound on COND(A). Default is 1e15.
%    show       boolean, 0 to suppress outputs, 1 to show iterations.
%               Default is 0. 
%    p1, p2,... Optional, inputs to A and M if they are functions
%
% OUTPUTS:
%    x       n-vector, estimated solution 
%    flag    integer, convergence flag
%           -1  beta2 = 0.  If M = I, b and x are eigenvectors.             
%            0 beta1 = 0.  The exact solution is  x = 0.                    
%            1 A solution to (poss. singular) Ax = b found, given rtol.
%            
%            2 Pseudoinverse solution for singular LS problem, given rtol.  
%            3 A solution to (poss. singular) Ax = b found, given eps.    
%            4 Pseudoinverse solution for singular LS problem, given eps.   
%            5 x has converged to an eigenvector.                         
%            6 xnorm has exceeded maxxnorm.                                
%            7 Acond has exceeded Acondlim.                                 
%            8 The iteration limit was reached.                             
%            9 It is a least squares problem but no converged solution yet.
%    iter    integer, iteration number at which x was computed: 0 <= iter <= maxit.
%    relres  real positive, the relative residual is defined as
%                 NORM(b-A*x)/(NORM(A) * NORM(x) + NORM(b)),
%            computed recurrently here.  If flag is 1 or 3,  relres <= TOL.
%    relAres real positive, the relative-NORM(Ar) := NORM(Ar) / NORM(A) ---
%            computed recurrently here. If flag is 2 or 4, relAres <= TOL. 
%    Anorm   real positive, estimate of matrix 2-norm of A.
%    Acond   real positive, estimate of condition number of A with
%            respect to 2-norm. 
%    xnorm   non-negative positive, recurrently computed NORM(x)
%    Axnorm  non-negative positive, recurrently computed NORM(A * x).
%
% EXAMPLE 1: 
%     n = 100; on = ones(n,1); A = spdiags([-2*on 4*on -2*on],-1:1,n,n);
%     b = sum(A,2); rtol = 1e-10; maxit = 50; M = spdiags(4*on,0,n,n);
%     x = minresQLP49(A, b, rtol, maxit, M);
%   
%     Use this matrix-vector product function
%        function y = afun(x,n)
%        y = 4 * x;
%        y(2:n) = y(2:n) - 2 * x(1:n-1);
%        y(1:n-1) = y(1:n-1) - 2 * x(2:n);
%     as input to minresQLP49
%        x1 = minresQLP49(@afun, b, rtol, maxit, M);
%
% EXAMPLE 2: A is Laplacian on a 50 by 05 grid, singular and indefinite.
%      n = 50; N = n^2; on=ones(n,1);   B = spdiags([on on on], -1:1, n, n);
%      A = sparse([],[],[],N,N,(3*n-2)^2);
%      for i=1:n 
%          A((i-1)*n+1:i*n,(i-1)*n+1:i*n) = B; 
%          if i*n+1 < n*n, A(i*n+1:(i+1)*n,(i-1)*n+1:i*n)=B; end; 
%          if (i-2)*n+1 > 0  A((i-2)*n+1:(i-1)*n,(i-1)*n+1:i*n)=B;  end; 
%      end
%      b = sum(A,2);   rtol = 1e-5;   maxxnorm = 1e2; 
%      shift = 0;   Acondlim = [];   show = 1;   M = [];  
%      x = minresQLP49( A, b, rtol, N, M, shift, maxxnorm, Acondlim, show);
%
% EXAMPLE 3: A is diagonal, singular and indefinite. 
%      h = 1;  a = -10; b = -a; n = 2*b/h + 1;   
%      A = spdiags((a:h:b)', 0, n, n);
%      b = ones(n,1);   rtol = 1e-6;   maxxnorm = 1e2; 
%      shift = 0;   Acondlim = [];   show = 1;   M = [];  
%      x = minresQLP49( A, b, rtol, N, M, shift, maxxnorm, Acondlim, show);
%
% TIPS:
%      if speed is a concern, set inputs show = 0, fignum = 0; do not ask
%      for vector outputs.
%
% SEE ALSO:
%    minresQLP, minresSOL, minresSOLs, minres, symmlq, lsqr, cgls
%    available as free software at 
%         http://www.stanford.edu/group/SOL/software.html
% 
% REFERENCES:
%    Sou-Cheng Choi's PhD Dissertation, Stanford University, 2006.
%    
% CURRENT / FUTURE RELEASES of minresQLP49:
%    Available at: http://www.stanford.edu/group/SOL/software.html
% 
% FUTURE WORK:
%    MM/DD/2006: ---
%
% MODIFICATION HISTORY:
% 26 Aug 2005: 1. Adapted MINRES-QLP.m from minresSOL41.m. Unlooped first
%                 three iterations.
%              2. Debugged for 3x3 examples.
% 22 Oct 2005: 1. Debugged for 4x4 examples.
% 01 Nov 2005: 1. Debugged for 5x5 and larger examples.
% 05 Nov 2005: 1. Condition numbers of A = ratio of maximum gramma_k to 
%                 minimum nonzero gama_k over all iterations k, since
%                 Stewart showed that these diagonal elements in L in QLP 
%                 factorization of A are good estimates of its singular
%                 values.
% 27 Jan 2006: 1. Debugged preconditioned-MINRESQLP.
% 24 Feb 2006: 1. Made minresQLP capable of doing single precision 
%                 calculations, meaning all intermediate matrices & vectors
%                 
%                 are single if input A is a single variable:
%                       x = minresQLP(single(A), single(b))
%              2. Updated Modification History
%              3. Formatted the whole file.
% 08 Mar 2006: 1. Rewrote minresxxxM so that M\x will be replaced by 
%                 R\(R'\x) with R as Cholesky factor of M and a persistent 
%                 variable so that CHOL being called only once throughout 
%                 all MINRES-B iterations.
% 22 Feb 2009:  If A = [1 1; 1 1], b = [1 -1]', then T_1 and T_2 = 0. 
%               it follows that Anorm = 0 and relative Arnorm = 0. 
%               So we want to estimate Anorm as ||A w||/||w|| where w
%               is randomly generated.
%
% KNOWN BUGS:
%    MM/DD/2006: ---
%
% NOTES:
%    MM/DD/2006: ---
%
% AUTHORS: Terrya Sou-Cheng Choi, ICME, Stanford University
%          Michael Saunders, SOL, Stanford University
%
% CREATION DATE: 05/12/2006
%
% COPYRIGHT NOTICE:
%
%    This is Copyrighted Material. The software is COPYRIGHTED by the  
%    original authors.
% 
% COPYLEFT NOTICE:
%
%     Permission is granted to make and distribute verbatim copies of this 
%     file, provided that the entire file is copied **together** as a 
%     **unit**.
% 
%     The purpose of this permission notice is to allow you to make copies 
%     of the software and distribute them to others, for free or for a fee, 
%     subject to the constraint that you maintain the entire file here 
%     as a unit.  This enables people you give the software to be aware 
%     of its origins, to ask questions of us by e-mail, to request  
%     improvements, obtain later releases, and so forth.
% 
%     If you seek permission to copy and distribute translations of this 
%     software into another language, please e-mail a specific request to
%     saunders@stanford.edu and choi@sccm.stanford.edu.
% 
%     If you seek permission to excerpt a **part** of the software library 
%     for example to appear in a scientific publication, please e-mail a 
%     specific request to saunders@stanford.edu and choi@sccm.stanford.edu.
% 
% COMMENTS? 
%
%     E-mail saunders@stanford.edu and choi@sccm.stanford.edu
%
format short e
clear minresxxxM  
%%  Check inputs and set default values.
if nargin < 2
   error('Not enough input parameters')
end
if nargin < 3  || ~exist('rtol','var') || isempty(rtol)
   rtol   = 1e-6;
end
if nargin < 4  || ~exist('maxit','var') || isempty(maxit)
   maxit  = 20;
end
if nargin < 5  || ~exist('M','var') ||isempty(M)
  precon = false;
else
  precon = true;
end
if nargin < 6  || ~exist('shift','var') ||isempty(shift)
  shift  = 0;
end
if nargin < 7  || ~exist('maxxnorm','var') || isempty(maxxnorm)
  maxxnorm = 1e7;
end
if nargin < 8  || ~exist('Acondlim','var') || isempty(Acondlim)
   Acondlim  =  0.1/eps;
end
if nargin < 9  || ~exist('show','var') || isempty(show)
   show   = true;
end
if nargin < 10  || ~exist('testsym','var') || isempty(testsym)
   testsym   = true;
end
if nargin < 11  || ~exist('fignum','var') || isempty(fignum)
   fignum = 0;
end
if nargin < 12 ||  ~exist('debug','var') ||isempty(debug)
   debug  = false;
end
if nargin < 13 ||  ~exist('disable','var') ||isempty(disable)
   disable  = false;
end
if nargin < 14 ||  ~exist('isbestiter','var') ||isempty(isbestiter)
   isbestiter  = 0;
end
if nargin < 15 ||  ~exist('isbestiter','var') ||isempty(isbestiter)
   isbestiter  = 0;
end

n      = length(b);
flag   = 0;
done   = false; 
if isa(A,'float')
   classname = class(A);
else
   classname = 'double';
end

%%  Check outputs:   
%     resvec,  Aresvec, xnormvec, pnormvec,  gamavec, alfavec, betavec, VV,...
%     resvecdirect, relresvec, Aresvecdirect, relAresvec,
%     xnormdirectvec, pnormdirectvec
if nargout > 9
   resvec         = zeros(maxit,1,classname);
else 
   resvec         = [];
end
if nargout > 10
   Aresvec   = zeros(maxit,1,classname);
else
   Aresvec   = [];
end
if nargout > 11
   xnormvec   = zeros(maxit,1,classname);
else
   xnormvec   = [];
end
if nargout > 12
   pnormvec   = zeros(maxit,1,classname);
else
   pnormvec   = [];
end
if nargout > 13
   gamavec = zeros(maxit,1);
else
   gamavec = [];
end
if nargout > 14
   veplnvec = zeros(maxit,1);
else
   veplnvec = [];
end
if nargout > 15
   etavec = zeros(maxit,1);
else
   etavec = [];
end
if nargout > 16
   alfavec = zeros(maxit,1);
else
   alfavec = [];
end
if nargout > 17
   betavec = zeros(maxit,1);
else
   betavec = [];
end
if (nargout > 18) || fignum
   VV   = zeros(n, maxit);
else
   VV = [];
end
if nargout > 19
   resvecdirect   = zeros(maxit,1,classname);
else
   resvecdirect   = [];
end
if nargout > 20
   relresvec   = zeros(maxit,1,classname);
else
   relresvec   = [];
end
if nargout > 21
   Aresvecdirect   = zeros(maxit,1,classname);
else
   Aresvecdirect   = [];
end
if nargout > 22
   relAresvec   = zeros(maxit,1,classname);
else
   relAresvec   = [];
end
if nargout > 23
   xnormdirectvec  = zeros(maxit,1,classname);
else
   xnormdirectvec   = [];
end
if nargout > 22
   pnormdirectvec = zeros(maxit,1,classname);
else
   pnormdirectvec = [];
end
if nargout > 24
   tknormvec = zeros(maxit,1,classname);
else
   tknormvec = [];
end
if isbestiter
   allvecs = zeros(maxit,1,classname);
else
   allvecs = [];
end

if fignum
   localOrthVtVmI  = zeros(maxit,1,classname);
   globalOrthVtVmI = zeros(maxit,1,classname);
end

%% Initalize

first = 'Enter minres-QLP49.   ';
last  = 'Exit  minres-QLP49.   ';
msg   =[' beta2 = 0.  If M = I, b and x are eigenvectors.             '   % -1
        ' beta1 = 0.  The exact solution is  x = 0.                   '   %  0
        ' A solution to (poss. singular) Ax = b found, given rtol.    '   %  1
        ' Pseudoinverse solution for singular LS problem, given rtol. '   %  2
        ' A solution to (poss. singular) Ax = b found, given eps.     '   %  3
        ' Pseudoinverse solution for singular LS problem, given eps.  '   %  4
        ' x has converged to an eigenvector.                          '   %  5
        ' xnorm has exceeded maxxnorm.                                '   %  6
        ' Acond has exceeded Acondlim.                                '   %  7
        ' The iteration limit was reached.                            '   %  8
        ' It is a least squares problem but no converged solution yet.'   %  9
        ' A null vector obtained, given rtol.                         ']; %  10
beta1 =  norm(b);
if show
   disp( ' ' )
   disp( [ first 'Pseudoinverse solution of symmetric Ax = b or min||Ax - b||.' ] )
   gprintf( sprintf( '\nn     = %3g  precon = %4g         shift   =  %10.2e     ||b||   =  %10.2e',...
                   n,           precon,              shift,               beta1) );
   gprintf( sprintf( '\nmaxit = %3g  rtol   = %11.2e  maxxnorm = %11.2e    Acondlim = %11.2e\n' ,...
                   maxit,       rtol ,           maxxnorm,            Acondlim) );
end
if isbestiter
   allvecs        = zeros(maxit,1,classname);
end
%---------------------------------------------------------------------
%% Decode A.
%---------------------------------------------------------------------
if isa(A,'float')         % A is an explicit matrix A, single or double
   [mA,nA] = size(A);
   if mA~=n || nA~=n
      error('A must be square & have the same dimension as length of b.')
   end
   if show,
      if issparse(A)
         fprintf('\nA is an explicit sparse matrix.')
      else
         fprintf('\nA is an explicit dense matrix.' )
      end
   end
elseif isa(A,'function_handle')
   disp(['The matrix A is defined by function_handle ' func2str(A) '.'])
else
   error('A must be a matrix or a function handle.')
end

%---------------------------------------------------------------------
%% Decode M.
%---------------------------------------------------------------------
if precon
   if isa(M,'float')       % M is an explicit matrix M.
      [mM,nM] = size(M);
      if mM~=n || nM~=n
         error('M must be square & have the same dimension as length of b.')
      end
      if show,
         if issparse(M)
            fprintf('\nM is an explicit sparse matrix.')
         else
            fprintf('\nM is an explicit dense matrix.' )
         end
      end
   elseif isa(M,'function_handle')
      disp(['The matrix M is defined by function_handle ' func2str(M) '.'])
   else
      error('M must be a matrix or a function handle.')
   end
end % if precon

%---------------------------------------------------------------------
%% Check if using single precision for variables and computations.
%---------------------------------------------------------------------
if isa(A, 'single') && isa(b, 'single')
   if precon && isa(M,'double'),
      M = single(M);
   end
   fprintf('\nComputations are done in single precision.')
end

% ------------------------------------------------------------------
%% Test A and M for symmetry.
% IsOpSym6 uses a statistical test.
% ------------------------------------------------------------------
if testsym
   [issym, Anorm] = IsOpSym6(A, n, varargin);   % See if A is symmetric.
   if ~issym
      if show, disp(' '); end
      error('The matrix or operator "A" appears to be unsymmetric');
   end

   if precon
     [issym, Mnorm] = IsOpSym6(M, n, varargin);   % See if M is symmetric.
     Mnorm = normest(M);
     if ~issym
        if show, disp(' '); end
        error('The preconditioner "M" appears to be unsymmetric');
     end
   end
end

%------------------------------------------------------------------
%% Some people accidentally input a sparse b(!).
%------------------------------------------------------------------
if issparse(b)
   if show, disp(' '); end
   warning('b is a sparse vector.  minresQLP will make it dense');
   b = full(b);
end

%------------------------------------------------------------------
%% Set up p and v for the first Lanczos vector v1.
% p  =  beta1 P' v1,  where  P = inv(C).
% v is really P' v1.
%------------------------------------------------------------------
r3    = b;
r2    = b;
if precon
   r3 = minresxxxM( M, b, varargin );
   if length(r3) ~= n
      error('Mfun must return a vector of the same length as b')
   end
   beta1 = r3'*b;
   %  Test for an indefinite preconditioner.
   %  If b = 0 exactly, stop with x = 0.
   if beta1 > 0,
      beta1 = sqrt( beta1 );     
   else
      error('The preconditioner "M" appears to be indefinite or singular.');
   end
end
if beta1==0  % b = 0, so x = 0
   flag = 0;   done = true;
end

%------------------------------------------------------------------
%% Initialize other quantities.
% Note that Anorm has been initialized by IsOpSym6.
% ------------------------------------------------------------------
iter     = 0;      beta    = 0;       phi      = beta1;           
betan    = beta1;  gmin    = 0;       cs       = -1;      sn       = 0;        
cr1      = -1;     sr1     = 0;       cr2      = -1;      sr2      = 0;        
dltan    = 0;      gama    = 0;       gamal    = 0;       eta      = 0;       
etal     = 0;      etal2   = 0;       vepln    = 0;       veplnl   = 0;       veplnl2= 0;             
taul     = 0;      tau     = 0;       Acond    = 1;       rnorm    = betan;          
pnorm    = 0;      xnorm   = 0;       Anorm    = 0;       xl2norm  = 0;       
ul3      = 0;      ul2     = 0;       ul       = 0;       lastiter = 0;       
bestiter = 0;      prnt2   = true;    Axnorm   = 0;       lines    = 1;
headlines = 30 * lines;               L_ek_norm = 0;      u = 0;       gamal2   = 0;
 
relres  = rnorm / (Anorm*xnorm + beta1);
rnormbest = rnorm;
BIG       = 1000;
xrArnormsbest = BIG * Anorm * rnorm^2;

w      = minresxxxA( A, b, varargin );

if length(w) ~= n
   error('Afun must return a vector of the same length as b')
end
Arnorm       = norm(w);

if fignum 
   mfig = 2;          nfig = 4;          figure(fignum);     clf;
end
if fignum || (nargout > 8)
   rnormdirect = beta1;   xnormdirect = 0;       pnormdirect = 0;
   Arnormdirectl = 0;  Arnormdirect = Arnorm;    Arnormbest = Arnorm; 
end

xl2 = zeros(n,1,classname);
x  = zeros(n,1,classname);
w  = zeros(n,1,classname); % reset w to 0
wl = zeros(n,1,classname);
if isbestiter
   xbest = x;
end

if show || debug
   disp(' ')
   disp(' ')
   head1 = '  iter  bestiter  x(1)   xnorm     rnorm     Arnorm   ';
   head2 = 'Compatible   LS    norm(A)  cond(A)  beta';  
   disp([head1 head2])
end

%---------------------------------------------------------------------
%% Main iteration 
% --------------------------------------------------------------------

while ~done && iter < maxit
   iter  = iter + 1;
   %-----------------------------------------------------------------
   %% Obtain quantities for the next Lanczos vector vk+1, k = 1, 2,...
   % The general iteration is similar to the case k = 1 with v0 = 0:
   %
   %   p1      = Operator * v1  -  beta1 * v0,
   %   alpha1  = v1'p1,
   %   q2      = p2  -  alpha1 * v1,
   %   beta2^2 = q2'q2,
   %   v2      = (1/beta2) q2.
   %
   % Again, p = betak P vk,  where  P = C**(-1).
   % .... more description needed.
   %-----------------------------------------------------------------
%% Lanczos
   betal    = beta;
   beta     = betan;
   v        = r3 * (1/beta);                  % If precon, v = s_k / beta k.
   r3       = minresxxxA( A,  v, varargin );  
   if shift ~= 0
      r3    = r3 - shift*v;                   % r3 = p_k (will be changed below).                                          % If precon, r3 = p_k / beta_k
   end
   if fignum || ~isempty(pnormdirectvec)
       pnormdirectl = pnormdirect;
       if ~precon
          pnormdirect  = norm(r3);
       else
          pnormdirect  = sqrt( r3' * minresxxxM( M, r3, varargin ));
       end
       if ~isempty(pnormdirectvec) 
          pnormdirectvec(iter) = pnormdirect;
       end
   end
   if ~isempty(VV) 
      VV(:,iter) = v;   % V will be overwritten while updating w below
   end
   if iter  > 1
      r3    = r3 - (beta/betal) * r1;    % r1 = beta_{k-1} v_{k-1}. If precon, r1 = z_k-1.
   end
   alfa     = real( r3' * v );           % alpha_k
   r3       = r3 - ( alfa/beta ) * r2;    % r2 = beta_k v_k, r3 = beta_{k+1} v_{k+1}.
                                         % If precon, r2 = z_k, r3 = z_k+1.                           
   r1       = r2;                        % If precon, r1 = z_k
   r2       = r3;                        % If precon, r2 = z_k+1
   if ~precon,
      betan = norm(r3);                  % beta_{k+1}
   else
      r3    = minresxxxM( M, r2, varargin );
      betan = r2' * r3;
      if betan > 0,
         betan = sqrt( betan );     
      else
         error('The preconditioner "M" appears to be indefinite or singular.');
      end
   end
   pnorml  = pnorm;
   if iter == 1        
      pnorm  = norm([alfa, betan]);
   else
      pnorm  = norm([beta, alfa, betan]);
   end
   if debug
      fprintf('\n\nLanczos iteration %d :\n', iter);
      fprintf('\n  v_%d     = ', iter );  fprintf('%13.3e ', v(1:min(n,5)) );
      fprintf('\n  r1_%d    = ', iter );  fprintf('%13.3e ', r1(1:min(n,5)) );
      fprintf('\n  r2_%d    = ', iter );  fprintf('%13.3e ', r2(1:min(n,5)) );
      fprintf('\n  r3_%d    = ', iter );  fprintf('%13.3e ', r3(1:min(n,5)) );
      fprintf('\n  alpha_%d = %13.3e, beta_%d = %13.3e, beta_%d = %13.3e ',...
                   iter, alfa, iter, beta, iter+1, betan );
   end

%-----------------------------------------------------------------
%% Apply previous left rotation Qk-1 to get
%   [dlta_k epln_{k+1} ] = [cs  sn][dlta_k    0      ]
%   [gbar_k  dlta_{k+1}]   [sn -cs][alfa_k beta_{k+1}].
%-----------------------------------------------------------------
   dbar   = dltan;
   dlta   = cs*dbar + sn*alfa;       % dlta1 = 0         dlta k
   gbar   = sn*dbar - cs*alfa;       % gbar 1 = alfa1     gbar k
   eplnn  =           sn*betan;      % epln2 = 0         epln k+1
   dltan  =         - cs*betan;      % dltan 2 = beta2   dltan k+1
   if debug
      fprintf('\n\nApply previous left rotation Q_{%d,%d}:\n', iter-1, iter');
      fprintf('\n  c_%d     = %13.3e, s_%d    = %13.3e', ...
                   iter-1, cs, iter-1, sn );
      fprintf('\n  dlta_%d = %13.3e, gbar_%d = %13.3e', ...
                   iter, dlta, iter, gbar );
      fprintf('\n  epln_%d = %13.3e, dbar_%d = %13.3e', ...
                   iter+1, eplnn, iter+1, dltan );
   end

%% Compute the current left plane rotation Qk
   gamal3 = gamal2;     
   gamal2 = gamal;
   gamal  = gama;
   [cs, sn, gama] = SymGivens2(gbar, betan); 
   gama_tmp = gama;
   taul2   = taul;
   taul    = tau;
   tau     = cs*phi;               % phik
   phi     = sn*phi;               % phibar_{k+1}
   Axnorm = norm([Axnorm  tau]);   % norm([tau_1,..., tau_k])
   if debug
      fprintf('\n\nCompute the current left plane rotation Q_{%d,%d}:\n', iter, iter+1);
      fprintf('\n  c_%d     = %13.3e, s_%d    = %13.3e ', iter, cs, iter, sn );
      fprintf('\n  tau_%d   = %13.3e, phi_%d  = %13.3e ', iter, tau, iter, phi);
      fprintf('\n  gama_%d = %13.3e ', iter, gama  );
   end

%%  Apply the previous right plane rotation P{k-2,k}
   if iter > 2
      veplnl2  = veplnl;
      etal2    = etal;
      etal     = eta;
      dlta_tmp = sr2*vepln - cr2*dlta;  % vepln = vepln from last iter
      veplnl   = cr2*vepln + sr2*dlta;
      dlta     = dlta_tmp;
      eta      =   sr2*gama;
      gama     = - cr2*gama;
      if debug
         fprintf('\n\nApply the previous right plane rotations P_{%d,%d}:\n', iter-2, iter);
         fprintf('\n  cr2_%d   = %13.3e, sr2_%d    = %13.3e', ...
                      iter, cr2, iter, sr2 );
         fprintf('\n  gama_%d = %13.3e, gama_%d  = %13.3e, gama_%d = %13.3e', ...
                      iter-2, gamal2, iter-1, gamal, iter, gama);
         fprintf('\n  dlta_%d = %13.3e, vepln_%d = %13.3e, eta_%d   = %13.3e', ...
                      iter, dlta, iter-1, veplnl, iter, eta );
      end
   end
   
%% Compute the current right plane rotation P{k-1,k}, P_12, P_23,...
   if iter > 1
      [cr1, sr1, gamal] = SymGivens2(gamal, dlta);
      vepln  =   sr1*gama;
      gama   = - cr1*gama;
      if debug
         fprintf('\n\nCompute the second current right plane rotations P_{%d,%d}:\n', iter-1, iter');
         fprintf('\n  cr1_%d   = %13.3e, sr1_%d   = %13.3e', ...
                      iter, cr1, iter, sr1 );
         fprintf('\n  gama_%d = %13.3e, gama_%d = %13.3e, vepln_%d = %13.3e',...
                      iter-1, gamal, iter, gama, iter, vepln );
      end
   end
   
%% Update xnorm
   
   xnorml  = xnorm;
   ul4   = ul3;
   ul3   = ul2;
   if iter > 2
      ul2   = ( taul2 - etal2 * ul4 - veplnl2 * ul3 ) / gamal2;
   end
   if iter > 1
      ul    = ( taul  - etal  * ul3 - veplnl  * ul2) / gamal;
   end
   xnorm_tmp = norm([xl2norm ul2 ul]);
   if (abs(gama) > eps) && (xnorm_tmp < maxxnorm)
      u = (tau - eta * ul2 - vepln * ul ) / gama; 
      if norm([xnorm_tmp u]) > maxxnorm
         u = 0;
         flag = 6;
      end
   else
      u    = 0;
      flag = 9;  
   end   

   xl2norm = norm([xl2norm ul2]);  
   xnorm   = norm([xl2norm ul u]); 
 


%% Update w. Update x  except if it will become too big
   if iter == 1
      wl2 = wl;
      wl  = v * sr1;
      w   = v * -cr1;   
   elseif iter == 2
      wl2 = wl;
      wl  = w * cr1  + v * sr1;   % w = v_1
      w   = w * sr1  + v * -cr1;
   else
      wl2 = wl;
      wl  = w;
      w   = wl2*sr2 - v*cr2;
      wl2 = wl2*cr2 + v*sr2;
      v   = wl *cr1 + w*sr1;    % temporarily using v for intermediate wl
      w   = wl *sr1 - w*cr1;
      wl  = v;  
   end
   x1last  = x(1);
   xl2     = xl2 + wl2 * ul2;  % wl2 = wl3;
   x       = xl2 + wl * ul + w * u;
   
   if debug
      fprintf('\n\nUpdate w:\n');
      fprintf('\n  w_%d     = ', iter-1 ); fprintf('%13.3e ', wl(1:min(n,5)) );
      fprintf('\n  w_%d     = ', iter ); fprintf('%13.3e ', w(1:min(n,5)) );
   end
      if debug
      fprintf('\n\nUpdate u, x and xnorm:\n');
      fprintf('\n  u_%d     = %13.3e, u_%d     = %13.3e, u_%d     = %13.3e',...
              iter-2, ul2, iter-1, ul, iter, u );
      fprintf('\n  x_%d     = ', iter ); fprintf('%13.3e ', x(1:min(n,5)) );
      fprintf('\n  ||x_%d|| = ', iter ); fprintf('%13.3e ', xnorm );
   end
   
%% Compute the next right plane rotation P{k-1,k+1}
   gamal_tmp = gamal;
   [cr2, sr2, gamal] = SymGivens2(gamal, eplnn);
   if debug
      fprintf('\n\nCompute the next right plane rotation P_{%d,%d}:\n', iter-1, iter+1);
      fprintf('\n  cr2_%d   = %13.3e, sr2_%d    = %13.3e,  gama_%d = %13.3e', ...
                   iter+1, cr2, iter+1, sr2, iter-1, gamal);
   end 
   

%% Estimate various norms
   rnorml    = rnorm;                          % ||r_{k-1}||
   Anorml    = Anorm;
   Acondl    = Acond;
   relresl   = relres;
   L_ek_norml = L_ek_norm;
   
   if flag ~= 9
      if iter > 1
         Tnorm = norm([Tnorm; beta; alfa; betan]); % Frobenius norm OF T_{k,k+1}
                                                   % used as estimate of AnormF.
      else 
         Tnorm = norm([alfa; betan]); 
      end
      AnormF   = norm(Tnorm);          
      L_el2_norm = norm([gamal2 veplnl eta]);
      Anorm = max([Anorm, pnorm, gamal]);
      rnorm    = phi;                           % ||r_k||
      relres = rnorm / (Anorm*xnorm + beta1);   % ||r||/(||A||||x||+||b||)
      if iter == 1
         gmin    = gama; 
      elseif iter > 1
         gmin    = min([gmin, gamal, abs(gama)]);
         %gmin    = min([gmin, gamal]);
      end
      Acond      = Anorm / gmin;
   end 
   
   if fignum && (iter > 1)
      Arnorml2    = Arnorml;
      relAresl2   = relAresl;
   end
   rootl       = norm([gbar; dltan]);
   Arnorml     = rnorml * rootl;           % ||A r_{k-1} ||
   relAresl = rootl / Anorm;               % ||Ar|| / (||A|| ||r||)     
   %relAresl = Arnorml / Anorm;             % ||Ar|| / ||A||
   
   if isbestiter,
      xrArnormsl = xnorml*rnorml*Arnorml;
      if xrArnormsbest > xrArnormsl
         bestiter      = iter - 1;
         xbest         = wl2;
         xnormbest     = xnorml; 
         rnormbest     = rnorml;
         Arnormbest    = Arnorml;
         relresbest    = relresl;
         relAresbest   = relAresl;
         if iter       > 1, xrArnormsbest = xrArnormsl; end
      end
   end
   
   if fignum || (nargout > 8) 
      rnormdirectl     = rnormdirect;
      rdirect          = b - minresxxxA( A,x,varargin ) + shift*x;
      if ~precon
         rnormdirect   = norm(rdirect);
      else
         Mrdirect      = minresxxxM( M, rdirect, varargin );
         rnormdirect   = sqrt( rdirect' * Mrdirect);
      end
      Arnormdirectl2   = Arnormdirectl;
      Arnormdirectl    = Arnormdirect;
      if ~precon
          Arnormdirect  = norm(minresxxxA( A, rdirect,varargin ) - shift * rdirect);
      else
          Mrdirect      = minresxxxA( A, Mrdirect,varargin ) - shift * Mrdirect;   
          Arnormdirect  = sqrt( Mrdirect' *  minresxxxM( M, Mrdirect, varargin ) );
      end
      xnormdirectl  = xnormdirect;
      if ~precon
         xnormdirect   = norm(x);       % rather expensive
      else
         if isa(M,'float')
            xnormdirect   =  sqrt( x' * M * x );
         else
            xnormdirect   = norm(x);    % approximate
         end
      end
      if ~isempty(resvec)
         resvec(iter) = rnorm;
      end
      if ~isempty(relresvec)
         relresvec(iter) = relres;
      end 
      if ~isempty(resvecdirect)
         resvecdirect(iter) = rnormdirect;
      end
      if (iter > 1) 
         if ~isempty(Aresvec) 
            Aresvec(iter-1) = Arnorml; 
         end
         if ~isempty(relAresvec) 
            relAresvec(iter-1) = relAresl;
         end
      end
      if ~isempty(Aresvecdirect) 
         Aresvecdirect(iter) = Arnormdirect;
      end
      if ~isempty(xnormvec)
         xnormvec(iter) = xnorm;
      end
      if ~isempty(xnormdirectvec)
         xnormdirectvec(iter) = xnormdirect;
      end
      if isbestiter
         allvecs(iter)     = xnorm * relres * relAresl;
      end
      if ~isempty(pnormvec)
         pnormvec(iter) = pnorm;
      end
      if ~isempty(pnormdirectvec)
         pnormdirectvec(iter) = pnormdirect;
      end
      if ~isempty(resvec)
         tknormvec(iter) = Axnorm;
      end
      if ~isempty(gamavec) 
         if iter > 2
               gamavec(iter-2)   = gamal2;
         end
         if iter > 1
            gamavec(iter-1)   = gamal;
            veplnvec(iter-1) = veplnl;
         end
         gamavec(iter)        = abs(gama);
         veplnvec(iter)       = vepln;
         etavec(iter)         = eta;
      end
      if ~isempty(alfavec) 
         alfavec(iter)        = alfa;
      end
      if ~isempty(betavec) 
         betavec(iter)        = betan;
      end
   end
   if debug
      fprintf('\n\nUpdate other norms:\n');
      fprintf('\n  gmin_%d   = ', iter );   fprintf('%13.3e ', gmin );
      fprintf('\n  Tnorm_%d  = ', iter );   fprintf('%13.3e ', Tnorm );
      fprintf('\n  rnorm_%d  = ', iter );   fprintf('%13.3e ', rnorm );
      fprintf('\n  Arnorm_%d = ', iter-1);  fprintf('%13.3e ', Arnorml);
      fprintf('\n  Acond_%d  = ', iter );   fprintf('%13.3e ', Acond );
      fprintf('\n\n');
   end

%---------------------------------------------------------------
%% See if any of the stopping criteria are satisfied.
% In rare cases, flag is already -1 from above (Abar = const*I).
%---------------------------------------------------------------
   %epsa   = Anorm * eps;             % Used below to test betan and ||A r|| --- never used?
   epsx   = Anorm * xnorm * eps;
   epsr   = Anorm * xnorm * rtol;  
   % Test for singular Tk (hence singular A)
   % or x is already an LS solution (so again A must be singular).

   if (flag==0) || (flag == 9)
      t1 = 1 + relres;             % These tests work if rtol < eps
      t2 = 1 + relAresl;
      if rtol   >= eps
         if     t1         <= 1       , flag = 3;  % Accurate Ax=b solution
         elseif t2         <= 1       , flag = 4;  % Accurate LS solution
         elseif relres     <= rtol    , flag = 1;  % Good enough Ax=b solution
         elseif relAresl   <= rtol    , flag = 2;  % Good enough LS solution
         elseif epsx       >= beta1   , flag = 5;  % x is an eigenvector
         elseif xnorm      >= maxxnorm, flag = 6;  % xnorm exceeded its limit
         elseif Acond      >= Acondlim, flag = 7;  % Huge Acond (but maybe ok?)
         elseif iter       >= maxit   , flag = 8;  % Too many itns
         end
      else 
         if     relres     <= rtol    , flag = 1;   % Good enough Ax=b solution
         elseif relAresl   <= rtol    , flag = 2;   % Good enough LS solution
         elseif epsx       >= beta1   , flag = 5;   % x is an eigenvector
         elseif xnorm      >= maxxnorm, flag = 6;   % xnorm exceeded its limit
         elseif Acond      >= Acondlim, flag = 7;   % Huge Acond (but maybe ok?)
         elseif iter       >= maxit   , flag = 8;   % Too many itns
         end
       end
   end
   if disable && (iter < maxit)
       flag = 0;
       done = 0;
       %Axnorm = norm(A*x); 
       %Axnorm - Axnorm
       if Axnorm < rtol * Anorm * xnorm
          flag = 10; 
          lastiter = 0;
       end
   end
   if flag ~= 0 % flag may have been changed above
      done      = 1;
      if precon
         xnorm = norm(x);  % So far, xnorm is an estimate of sqrt(x'Mx)
      end
      r1 = b - minresxxxA( A, x , varargin ) + shift*x;  
          
      relres = rnorm / (Anorm * xnorm + beta1);
      Arnorm = norm( minresxxxA( A, r1, varargin ) - shift * r1 );  
      if rnorm > realmin
       relAres = Arnorm / (Anorm * rnorm);
      end
      if (flag == 2) || (flag == 4) || (flag == 7) 
         % LS, or maxxnorm exceeded, or Acondlim exceeded
         lastiter = 1;
      end
      if  lastiter  || (flag == 6)
          rnorm = rnorml;    relres = relresl;
          if lastiter
             iter      = iter - 1; 
             prnt2     = false;
             pnorm     = pnorml; 
          end
          if fignum && iter > 1 
             relAresl = relAresl2;
             Arnorml   = Arnorml2;
             Arnormdirect = Arnormdirectl;
             Arnormdirectl = Arnormdirectl2;
          end
          gama      = gamal;
          gamal     = gamal2;    
      end   
      if ~isempty(Aresvec)  & iter>0  
            Aresvec(iter) = Arnorm;
      end
      if ~isempty(relAresvec) & iter>0 
            relAresvec(iter) = relAres;
      end
      if flag == 9
         prnt2 = true;
      end
   end
%% See if it is time to print something.
   prnt = false;
   if n            <= 40       , prnt = true; end
   if iter         <= 10       , prnt = true; end
   if iter         >= maxit-10 , prnt = true; end
%    if rnorm        <= 10*epsx  , prnt = true; end
%    if rnorm        <= 10*epsr  , prnt = true; end
%    if Acond        <= 1e-2/eps , prnt = true; end
  
   if mod(iter-1,lines) ==0        , prnt = true; end
   if flag         ~= 0        , prnt = true; end

   if debug
      disp([head1 head2]);
   end
   
   if show && prnt && prnt2 
      str1 = sprintf( '\n%6g %6g %10.2e %10.2e %10.2e %10.2e %10.3e', ...
             iter-1, bestiter, x1last, xnorml, rnorml, Arnorml, relresl );
      str2 = sprintf( ' %10.3e', relAresl);
      str3 = sprintf( ' %8.1e %8.1e %8.1e ', Anorml, Acondl, betal);
      gprintf([str1 str2 str3]);
 
    if iter == 101     
       lines = 10;     headlines = 30 * lines;
    elseif iter == 1001     
       lines = 100;     headlines = 30 * lines;   end
    if mod(iter, headlines) == 0     
       fprintf('\n\n%s\n', [head1 head2]);     
    end
 
   end % show && prnt


%% plot figures
   if fignum && prnt && (iter > 1) && ~lastiter && prnt2
      iterl = iter-1;
      subplot(mfig,nfig,1);
      semilogy(iterl, max(rnorml, realmin), 'r.');   hold on; 
      semilogy(iterl, max(relresl, realmin), 'k.');
 
      subplot(mfig,nfig,2);
      semilogy(iterl, max(Arnorml, realmin), 'r.');   hold on;
      semilogy(iterl, max(relAresl, realmin), 'k.');

      subplot(mfig,nfig,3);
      semilogy(iterl, max(xnorml,realmin), 'r.');   hold on;
      
      subplot(mfig,nfig,4);
      plot(iterl, pnorml  , 'r.');  hold on;
    
      subplot(mfig,nfig,5);
      semilogy(iterl, gamal2, 'r.');   hold on;  
      semilogy(iterl, Anorml, 'b.');
      semilogy(iterl, Acondl, 'k.');
      semilogy(iterl, L_el2_norm, 'gs');   hold on;
      
      subplot(mfig,nfig,6);
      loglog(xnorml, rnorml, 'r.'); hold on;
   end
end % main loop
 
%% Post-processing
 
xnormdirect   = norm(x);       % rather expensive
      
if isbestiter,
   if xrArnormsbest >= xnorm*rnorm*Arnorm
      bestiter      = iter;
      xbest         = x;  
      xnormbest     = xnorm; 
      rnormbest     = rnorm;
      Arnormbest    = Arnorm;
      relresbest  = relres;
      relAresbest = relAres;
   end
   x         = xbest;
   xnorm     = xnormbest;
   rnorm     = rnormbest;
   Arnorm    = Arnormbest;
   relres  = relresbest;
   relAres = relAresbest;
   iter      = bestiter;
end
 
%% print final iteration
if show && prnt   
   str1 = sprintf('\n%6g %6g %10.2e %10.2e %10.2e %10.2e %10.3e ', ...
		              iter, bestiter, x(1), xnorm, rnorm, Arnorm, relres);
   if rnorm > realmin
      str2 = sprintf( '%10.3e %8.1e %8.1e %8.1e ', relAres, Anorm, Acond, beta);
   else 
      str2 = sprintf( '          %8.1e %8.1e %8.1e ',  Anorm, Acond, beta);
   end
   str3 = sprintf( '\n%98.1e                 ',                betan);
   gprintf([str1 str2 str3]);
end % show && prnt

%% Display final status.
if debug
   fprintf('\n\n');
end
if show
   r1 = b - minresxxxA( A,x,varargin ) + shift*x; % use r1 to temporarily store residual vector
   disp( ' ' )
   gprintf( [last sprintf( ' flag    =  %2g               iter =%4g\n', ...
                          flag,                 iter   )] );
   gprintf( [last sprintf( ' Anorm   =  %12.4e     Acond =  %12.4e\n' , ...
                          Anorm,                Acond  )] );
   gprintf( [last sprintf( ' rnorm   =  %12.4e     rnorm direct  =  %12.4e\n' ,...
                          rnorm,                norm(r1)  )] );
   gprintf( [last sprintf( ' Arnorm  =  %12.4e     Arnorm direct =  %12.4e\n',...
                          Arnorm,              norm(minresxxxA( A,r1,varargin )-shift*r1)  )] );
   gprintf( [last sprintf( ' xnorm   =  %12.4e     xnorm direct  =  %12.4e\n' ,...
                          xnorm,               norm(x) )] );
   gprintf( [last msg(flag+2,:)]);
   disp( ' ' )
end

  
%% Plot residual norm, solution norm and norm of A r_k
if fignum
   VV = VV(:,1:iter);
   VtVmI = VV' * VV - speye(iter,iter);
   localOrthVtVmI(1)  = eps; % v_1 orthogonal to v_0=0
   globalOrthVtVmI(1) = eps;
   for i = 3:iter  % v_i v_i-1 v_i-2
       if i > 2
          localOrthVtVmI(i)  = norm(VtVmI(i-2:i,i-2:i),'inf');
       else
          localOrthVtVmI(i)  = norm(VtVmI(1:2,1:2),'inf');
       end
       globalOrthVtVmI(i) = norm(VtVmI(1:i,1:i),'inf');
   end
   localOrthVtVmI  = localOrthVtVmI(1:iter-1);
   globalOrthVtVmI = globalOrthVtVmI(1:iter-1);
end

if  ~isempty(resvec) 
   resvec      = resvec(1:iter); 
end
if ~isempty(resvecdirect) 
   resvecdirect= resvecdirect(1:iter);
end
if ~isempty(relresvec) 
   relresvec   = relresvec(1:iter);
end
if ~isempty(Aresvec) 
   Aresvec     = Aresvec(1:iter);
end
if ~isempty(relAresvec) 
   relAresvec  = relAresvec(1:iter);
end
if ~isempty(Aresvecdirect)
   Aresvecdirect= Aresvecdirect(1:iter);
end
if ~isempty(xnormvec)
   xnormvec    = xnormvec(1:iter);
end
if ~isempty(xnormdirectvec)
   xnormdirectvec = xnormdirectvec(1:iter);
end
if isbestiter
   allvecs     = allvecs(1:iter);
end
if ~isempty(gamavec)   
   gamavec     = gamavec(1:iter);
end
if ~isempty(veplnvec)   
   veplnvec     = veplnvec(1:iter);
end
if ~isempty(etavec)   
   etavec     = etavec(1:iter);
end
if ~isempty(alfavec)    
   alfavec     = alfavec(1:iter);
end
if ~isempty(betavec)  
   betavec     = betavec(1:iter);
end
if ~isempty(pnormvec) 
   pnormvec    = pnormvec(1:iter);
end
if ~isempty(pnormdirectvec) 
   pnormdirectvec  = pnormdirectvec(1:iter);
end
if ~isempty(tknormvec) 
   tknormvec       = tknormvec(1:iter);
end
 
if fignum && iter > 1
   h = gcf;
   set(h, 'Position', [275   300   860   645]);
   options={'Interpreter','latex'}; 
   % plot last iteration, legends and axis labels
   h = subplot(mfig,nfig,1);
   semilogy(iter, max(rnorm, realmin), 'r.');   
   semilogy(iter, max(relres, realmin), 'k.')
   hl = legend('$\phi_k$', '$\phi_k/(\|A\|_2^{(k)} \chi_k +\|b\|_2)$', 'Location', 'SouthOutside' ); 
   set(hl, 'Interpreter','latex');
   title('MINRES-QLP49');
   xlabel('Iteration $k$', options{:});  
   semilogy([1 iter], [rtol rtol], '-g'); 
   semilogy(iter, max(rnormdirect, realmin), 'rs'); 
   semilogy(iter, max(rnormdirect/(Anorm*xnormdirect+beta1), realmin), 'ks')
   axis tight; 
   axis square;
   set(h, 'Position', [1.3000e-001  5.8384e-001  1.5327e-001  3.4116e-001]); % x y width height
   
   h = subplot(mfig,nfig,2);
   semilogy(iter, max(Arnorm, realmin), 'r.');   
   semilogy(iter, max(relAres, realmin), 'k.');
   hl =  legend('$ \psi_k $','$ \psi_k / (\|A\|_2^{(k)} \phi_k) $', 'Location', 'SouthOutside'); 
   set(hl, 'Interpreter', 'latex');
   xlabel('Iteration $k$', options{:});  
   semilogy([1 iter], [rtol rtol], '-g');
   semilogy(iter, max(Arnormdirect, realmin), 'rs'); 
   semilogy(iter, max(Arnormdirect/(Anorm*rnormdirect), realmin), 'ks')
   axis tight;
   axis square;
   set(h, 'Position', [3.4257e-001  5.8384e-001  1.5327e-001  3.4116e-001]);
   
   h = subplot(mfig,nfig,3);
   semilogy(iter, max(xnorm,realmin), 'r.');   
   semilogy(iter, max(xnormdirect,realmin), 'rs'); 
   title('$\chi_k \simeq \|x_k\|_2$', options{:});  
   xlabel('Iteration $k$', options{:});  
   axis tight
   axis square;
   set(h, 'Position', [5.5566e-001  5.8384e-001  1.5327e-001  3.4116e-001]);
   
   h = subplot(mfig,nfig,4);
   plot(iter, pnorm  , 'r.');   
   plot(iter, pnormdirect, 'rs');   
   title('$\|T_k e_k\|_2 \simeq \|p_k\|_2$', options{:});
   xlabel('Iteration $k$', options{:});
   axis tight;
   axis square;
   set(h, 'Position', [7.4835e-001  5.8384e-001  1.5327e-001  3.4116e-001]);
   
   h = subplot(mfig,nfig,5);
   semilogy(iter, gamal,  'r.');   
   semilogy(iter, Anorml, 'b.');
   semilogy(iter, Acondl, 'k.');
   semilogy(iter, L_ek_norml, 'gs');
   hl = legend('$\gamma_k$','$\|A\|_2$','$\kappa_2(A)$', '$\|L\|_2$', 'Location', 'SouthOutside');
   set(hl, 'Interpreter','latex');
   xlabel('Iteration $k$', options{:});  
   axis tight;
   axis square; 
   set(h, 'Position', [1.3000e-001  1.5521e-001   1.5327e-001  3.4116e-001]);
   
   h = subplot(mfig,nfig,6);
   loglog(xnorm, rnorm, 'r.');  
   loglog(xnormdirect, rnormdirect, 'rs'); 
   ylabel('$\phi_k$', options{:});
   xlabel('$\chi_k$', options{:});  
   axis tight;
   axis square;
   set(h, 'Position', [3.7745e-001  1.5521e-001   1.5327e-001  3.4116e-001]);
   
   h = subplot(mfig,nfig,7);
   semilogy(1:iter-1,  max(localOrthVtVmI, eps), '-r');  hold on; 
   semilogy(1:iter-1, max(globalOrthVtVmI, eps), '--b');   hold off;
   title('$\| V_k^T V_k - I \|_\infty$', options{:});  
   hl = legend('local','global', 'Location', 'SouthOutside');
   set(hl, 'Interpreter','latex');
   xlabel('Iteration $k$', options{:});    
   axis tight;
   axis square;
   set(h, 'Position', [5.4869e-001  1.5521e-001   1.5327e-001  3.4116e-001]);
   
   h = subplot(mfig,nfig,8);
   cspy3(VtVmI(1:iter-1, 1:iter-1), 0, 5);
   title('CSPY( $| V_k^T V_k - I |$ )', options{:});
   xlabel('$k$', options{:});
   ylabel('$k$', options{:});
   axis tight;
   axis square;
   set(h, 'Position', [7.4835e-001  1.5521e-001   1.5327e-001  3.4116e-001]);
end
 
warning on;
clear minresxxxM  

%-----------------------------------------------------------------------
%% End function minresQLP49.m
%-----------------------------------------------------------------------
%% private functions

function p = minresxxxA( A,x,varargin )

%        p = minresxxxA( A,x,varargin )
% computes p = Ax for a matrix or operator A defined by input parameter A.

%-----------------------------------------------------------------------
% 17 Oct 2003: Default A*x function for minres.m.
%              User parameters A, varargin are passed thru to here.
%-----------------------------------------------------------------------

  if isa(A,'function_handle')
     p = A( x,varargin );
  else
     p = A*x;
  end

%-----------------------------------------------------------------------
% End private function minresxxxA
%-----------------------------------------------------------------------

function p = minresxxxM( M,x,varargin )

%        p = minresxxxM( M,x,varargin )
% solves My = x for a matrix or operator M defined by input parameter M.

%-----------------------------------------------------------------------
% 17 Oct 2003: Default M\x for minres.m.
%              User parameters M, varargin are passed thru to here.
% Precondition: M symmetric positive definite.
% Note: CHOL works with sparse matrices.
%-----------------------------------------------------------------------
 
% persistent R 
%  
% if isa(M,'function_handle')
%    p = M( x,varargin );
% else
%    %p = M\x; % calls CHOL in each iteration!!!
%    %The following calls CHOL only once throughout all iterations
%    if ~exist('R','var') || isempty(R)
%       % clear minresxxxM before existing MINRES-B.
%       R = chol(M); 
%    end 
%    p = R' \ x;
%    p = R  \ p;
% end
if isa(M,'function_handle')
   p = M( x,varargin );
else
   %p = M\x; % calls CHOL in each iteration!!!
   %The following calls CHOL only once throughout all iterations
   if ~exist('R','var')
      persistent R % clear minresxxxM before existing MINRES-B.
      R = chol(M); 
   end 
   p = R' \ x;
   p = R  \ p;
end

%-----------------------------------------------------------------------
% End private function minresxxxM
%-----------------------------------------------------------------------

function str = gprintf(str);
%-----------------------------------------------------------------------
%    format short e; a=.001234;
%    fprintf(regexprep(sprintf('%8.2E',a),'E-0','E-')) 
% gives 1.23E-03
%-----------------------------------------------------------------------
str = regexprep(str,'+0','+');
str = regexprep(str,'-0','-');
fprintf(str);
%-----------------------------------------------------------------------
% End private function gprintf
%-----------------------------------------------------------------------
