function [x, istop, rnorm, itn, resvec] = SQMR1(A, b, tol, itnlim, show);
% FILE_NAME: SQMR1.m
%  
% USAGE:
%     B = filename(A)
%
% DESCRIPTION:
%     Sovles symmetric system Ax = b using SQMR
%
% INPUTS:
%  A       nxn symmetric matrix
%  b       n-vector, right-hand side
%  tol     scalar, tolerance for stopping conditions
%  itnlim  integer, maximum number of iterations
%  show    show iteration details
%
% OUTPUTS:
%  x       n-vector, solution
%  istop   integer, stopping condition
%  rnorm   scalar, residual norm
%  itn     integer, number of iterations
%  resvec  vector, residual norms 
%
% EXAMPLE:
%    x = SQMR1(A, b);
%
% SEE ALSO:
%    ---
%
% REFERENCES:
%    Freund and Nachtigal, 1994
%
% FUTURE WORK:
%    MM/DD/2009: ---
%
% MODIFICATION HISTORY:
%    07/18/2006: No preconditioners as input in this version.
%
% KNOWN BUGS:
%    MM/DD/2009: ---
%
% NOTES:
%    MM/DD/2006: ---
%
% AUTHORS: Sou-Cheng (Terrya) Choi, ICME, Stanford University
%          Michael Saunders, SOL / ICME, Stanford University
%
% CREATION DATE: 07/18/2006
%
x    = 0;
r    = b;
t    = r;
tau  = norm(t);
beta1 = tau;
q    = t;
nu   = 0;
rho  = r' * q;
n    = length(b);
d    = zeros(n,1); % my guess
itn  = 1;
done = 0;
if nargout > 4
   resvec = zeros(itnlim,1);
else 
   resvec = [];
end
Anorm = normest(A);

while ~done
      t     = A*q;
      sigma = q'*t;
      if abs(sigma) < eps
         istop = 1;
         break;
      else
         alfa = rho/sigma;
         r    = r - alfa * t;
      end
      t    = r;
      taul = tau;
      tau  = norm(t);
      nul  = nu;
      nu   = tau / taul;
      c    = 1 / sqrt(1+nu^2);
      tai  = taul * nu * c;
      d    = (c * nul)^2 * d + c^2 * alfa * q;
      xl   = x;
      x    = x + d;
      xnorm = norm(x);
      if norm(x - xl) < tol
         istop = 2;
         break;
      end
      rnorm = sqrt(rho);
      if ~isempty(resvec)
         resvec(itn) = rnorm;
      end
      if rnorm  < tol * (Anorm * xnorm + beta1)
         istop = 3;
         break;
      else
         u    = t;
         rhol = rho;
         rho  = r'*u;
         beta = rho / rhol;
         q    = u +  beta * q;
      end 
      if show
         str1 = sprintf( '\n%6g  %10.2e %10.2e %10.2e ', ...
                itn, x(1), xnorm, rnorm );
         gprintf([str1]);
      end
      itn  = itn + 1;
      if itn >= itnlim
         istop = 4;
         break;
      end
end

if ~isempty(resvec)
   resvec = resvec(1:itn);
end 

if show
   str1 = sprintf( '\n%6g  %10.2e %10.2e %10.2e ', ...
          itn, x(1), xnorm, rnorm );
   gprintf([str1]);
end
      