 function [ x, flag, iter, rnorm, Anorm, Acond, xnorm, ...
           resvec, Aresvec, xnormvec, bestiter] = ...
  symmlqSOL3( A, b, rtol, maxit, maxxnorm, M, shift, show, fignum, ...
               debug, disable, isbestiter, varargin )

%         [ x, flag, iter, rnorm,  Anorm, Acond, xnorm, ...
%           resvec, Aresvec, xnormvec, bestiter] = ...
%  symmlqSOL3( A, b, rtol, maxit, maxxnorm, M, shift, show, fignum, ...
%                debug, isbestiter, varargin )
%  SYMMLQ is designed to solve the system of linear equations Ax = b
%  where A is an n by n symmetric matrix and b is a given vector.
%  A is accessed by means of a function call of the form
%               y = aprod ( n, x, varargin )
%  which must return the product y = Ax for any given vector x.
%  A positive-definite preconditioner M = C C' may optionally
%  be specified.  If precon is true, a function call of the form
%               y = msolve( n, x, varargin )
%  must solve the system My = x.
%  WARNING:   The files containing the functions 'aprod' and 'msolve'
%             must not be called A.m or M.m !!!!
%
%  For further information, type    help symdoc.

%  07 Jun 1989: Date of Fortran 77 version written by
%               Michael Saunders, Stanford University.
%  15 May 1990: MATLAB m-file symmlq.m derived from Fortran version
%               by Gerald N. Miranda Jr, UCSD.
%  02 Oct 1997: Move to CG point only if it is better than LQ point.
%               For interest, print qrnorm (= rnorm for minres).
%               Note that cgnorm is always one step ahead of qrnorm.
%  20 Oct 1999: Bug.  alfa1 = 0 caused Anorm = 0, divide by zero.
%               Need to estimate Anorm from column of Tk.
%  ------------------------------------------------------------------
%

%%  Check inputs and set default values.
if nargin < 2
  error('Not enough input parameters')
end
if nargin < 3  || isempty(rtol)
  rtol   = 1e-6;
end
if nargin < 4  || isempty(maxit)
  maxit  = 20;
end
if nargin < 5  || isempty(maxxnorm)
  maxxnorm = 1e5;
end
if nargin < 6  || isempty(M)
  precon = false;
else
  precon = true;
end
if nargin < 7  || isempty(shift)
  shift  = 0;
end
if nargin < 8  || isempty(show)
  show   = true;
end
if nargin < 9  || isempty(fignum)
  fignum = 0;
end
if nargin < 10 || isempty(debug)
   debug = false;
end
if nargin < 11 ||  ~exist('disable','var') ||isempty(disable)
   disable  = false;
end
if nargin < 12 || isempty(isbestiter)
   isbestiter  = 0;
   bestiter    = 0;
end

%%  Initialize 
n     = length(A);
first = 'Enter SYMMLQ.   ';
last  = 'Exit  SYMMLQ.   ';
space = ' ';
msg   =[' beta2 = 0.  If M = I, b and x are eigenvectors.    '   % 1
        ' beta1 = 0.  The exact solution is  x = 0.          '   % 2 
        ' Requested accuracy achieved, as determined by rtol.'   % 3
        ' Reasonable accuracy achieved, given eps.           '   % 4
        ' x has converged to an eigenvector.                 '   % 5
        ' xnorm has exceeded maxxnorm.                       '   % 6
        ' Acond has exceeded 0.1/eps.                        '   % 7
        ' The iteration limit was reached.                   '   % 8
        ' A does not define a symmetric matrix.              '   % 9
        ' M does not define a symmetric matrix.              '   % 10
        ' M does not define a pos-def preconditioner.        '   % 11
        ' A null vector obtained, given rtol.                ']; % 12 
% true   = 1;   false  = 0;
 
if show
   disp( space );
   disp( [ first 'Solution of symmetric Ax = b' ] );
   gprintf( sprintf( 'n      =  %3g     precon =  %4g           shift  =  %11.3e',...
                   n,                precon,                 shift ) );
   gprintf( sprintf( '\nmaxit  =  %3g     eps    =  %11.2e    rtol   =  %11.3e\n' ,...
                   maxit,            eps,                rtol ) );
end

if fignum || (nargout > 7)
   resvec      = zeros(maxit,1);
   Aresvec     = zeros(maxit,1);
   xnormvec    = zeros(maxit,1);
   xnormvec = zeros(maxit,1);
   if isa(A,'single')
      resvec      = single(resvec);
      Aresvec     = single(Aresvec);
      xnormvec    = single(xnormvec);
      xnormvec = single(xnormvec); 
   end
else
   resvec   = [];
   Aresvec  = [];
   xnormvec = [];
end

%---------------------------------------------------------------------
%% Decode A.
%---------------------------------------------------------------------
if isa(A,'float')         % A is an explicit matrix A, single or double
   [mA,nA] = size(A);
   if mA~=n || nA~=n
      error('A must be square & have the same dimension as length of b.')
   end
   if show,
      if issparse(A)
         fprintf('\nA is an explicit sparse matrix.\n')
      else
         fprintf('\nA is an explicit dense matrix.\n' )
      end
   end
elseif isa(A,'function_handle')
   disp(['The matrix A is defined by function_handle ' func2str(A) '.'])
else
   error('A must be a matrix or a function handle.')
end

%---------------------------------------------------------------------
%% Decode M.
%---------------------------------------------------------------------
if precon
   if isa(M,'float')       % M is an explicit matrix M.
      [mM,nM] = size(M);
      if mM~=n || nM~=n
         error('M must be square & have the same dimension as length of b.')
      end
      if show,
         if issparse(M)
            fprintf('\nM is an explicit sparse matrix.\n')
         else
            fprintf('\nM is an explicit dense matrix.\n' )
         end
      end
   elseif isa(M,'function_handle')
      disp(['The matrix M is defined by function_handle ' func2str(M) '.'])
   else
      error('M must be a matrix or a function handle.')
   end
end % if precon


%---------------------------------------------------------------------
%% Check if using single precision for variables and computations.
%---------------------------------------------------------------------
if isa(A, 'single') && isa(b, 'single')
   if precon && isa(M,'double'),
      M = single(M);
   end
   fprintf('\nComputations are done in single precision.')
end

% ------------------------------------------------------------------
%% Test A and M for symmetry.
% IsOpSym6 uses a statistical test.
% ------------------------------------------------------------------
[issym, Anorm] = IsOpSym6(A, n, varargin);   % See if A is symmetric.
if ~issym
   if show, disp(' '); end
   error('The matrix or operator "A" appears to be unsymmetric');
end

if precon
   [issym, Mnorm] = IsOpSym6(M, n, varargin);   % See if M is symmetric.
   if ~issym
      if show, disp(' '); end
      error('The preconditioner "M" appears to be unsymmetric');
   end
end

%------------------------------------------------------------------
%% Some people accidentally input a sparse b(!).
%------------------------------------------------------------------
if issparse(b)
   if show, disp(' '); end
   warning('b is a sparse vector.  MINRES will make it dense');
   b = full(b);
end

%% Initialize other quantities.
flag   = 0;     ynorm  = 0;    w = zeros(n,1);  Acond = 0;
iter   = 0;     xnorm  = 0;    x = zeros(n,1);  done  = false;
Anorm  = 0;     rnorm  = 0;    v = zeros(n,1);

if isa(A,'single')
   x   = single(x);
   w   = single(w);
   v   = single(v);
end     

%%  Set up y for the first Lanczos vector v1.
%  y is really beta1 * P * v1  where  P = C^(-1).
%  y and beta1 will be zero if b = 0.

r3    = b;
r2    = b;
r1    = b;

if ~precon
   beta1 = norm(b);
else
   r3 = symmlqxxxM( M, b, varargin );
   if length(r3) ~= n
      error('Mfun must return a vector of the same length as b')
   end
   beta1 = r3'*b;
   %  Test for an indefinite preconditioner.
   %  If b = 0 exactly, stop with x = 0.
   if beta1 < 0
      error('The preconditioner "M" appears to be indefinite');
   else
      beta1  = sqrt( beta1 );
   end
end

if beta1==0  % b = 0, so x = 0
   flag = 0;  show = true;  done = true;
end
% 
% y      = b;     r1     = b;
% if precon,      y      = symlqxxxM(M, r1, varargin );  end
% b1     = y(1);  beta1  = r1' * y;

%  See if msolve is symmetric.

% if check  &  precon
%    r2     = symlqxxxM(M, y, varargin );
%    s      = y' * y;
%    t      = r1' * r2;
%    z      = abs( s - t );
%    epsa   = (s + eps) * eps^(1/3);
%    if z > epsa, flag = 8;  show = true;  done = true; end
% end
                                                
%  Test for an indefinite preconditioner.
%  If b = 0 exactly, stop with x = 0.

% if beta1 <  0, flag = 9;   show = true;  done = true;  end
% if beta1 == 0,             show = true;  done = true;  end

%  Here and later, v is really P * (the Lanczos v).

% if beta1 > 0
%   beta1  = sqrt( beta1 );
%   s      = 1 / beta1;
%   v      = s * y;
% 
%   %  See if aprod is symmetric.
% 
%   y = symmlqxxxA( A, v, varargin );
%   if check
%      r2   = symmlqxxxA( A, y, varargin );
%      s    = y' * y;
%      t    = v' * r2;
%      z    = abs( s - t );
%      epsa = (s + eps) * eps^(1/3);
%      if z > epsa, flag = 7;  done  = true;  show = true;  end
%   end

%%  Set up y for the second Lanczos vector.
%  Again, y is beta * P * v2  where  P = C^(-1).
%  y and beta will be zero or very small if Abar = I or constant * I.
% 
%   y    = (- shift) * v + y;
%   alfa = v' * y;
%   y    = (- alfa / beta1) * r1 + y;
% 
%%  Make sure  r2  will be orthogonal to the first  v.

%   z  = v' * y;
%   s  = v' * v;
%   y  = (- z / s) * v + y;
%   r2 = y;
% 
%   if precon,  y = symlqxxxM(M, r2, varargin );  end
%   betal   = beta1;
%   beta   = r2' * y;
%   if beta <= 0,
%      flag = 9; show = true;  done = true; 
%   else  
%      y = y*(1/beta);
%   end
  
  %  Cause termination (later) if beta is essentially zero.

%   beta  = sqrt( beta );
%   if beta <= eps, flag = -1; end

%%  See if the local reorthogonalization achieved anything.

%   denom = sqrt( s ) * norm( r2 )  +  eps;
%   s     = z / denom;
%   t     = v' * r2;
%   t     = t / denom;
% 
%   if show
%      disp( space );
%      disp( sprintf( 'beta1 =  %10.2e   alpha1 =  %9.2e', beta1, alfa ) );
%      disp( sprintf( '(v1, v2) before and after  %14.2e', s ) );
%      disp( sprintf( 'local reorthogonalization  %14.2e\n', t ) );
%   end

%%  Initialize other quantities.
  beta   = 0;         betan  = beta1;   rnorml = beta1;   relrnorml = 1;
  cgnorm = beta1;     varphin   = 0;       Tnorm  = 0;%alfa^2 + beta^2;
  %gbar   = alfa;      bstep  = 0;       ynorm2 = 0;
  dbarn  = 0;         snprod = 1;       %gmax   =  abs( alfa ) + eps;
  varphi   = beta1;     x1cg   = 0;       %gmin   = gmax;
  qrnorm = beta1;     prnt   = false;   gbar   = 0;        pnorm = 0;  
  cs     = -1;         sn     = 0;      csl    = -1;         snl    = 0;
  epslnn = 0;         epsr    = 0;      zl     = 0;          z      = 0;
%end  % if beta1 > 0
  str5 = ''; 
  
if show
  head1 = '   Itn  x(1)      ||x||   ||r||:  cg   symmlq   minres';
  head2 = '    relrnrom  Anorm   Acond   beta';
  disp( [head1 head2] )
                    
%  str1 = sprintf ( '%6g %12.5e %10.3e', iter, x1cg, cgnorm );
%  str2 = sprintf ( ' %10.3e  %8.1e',    qrnorm, bstep/beta1 );
%  disp( [str1 str2] )
end

%  ------------------------------------------------------------------
%%  Main iteration loop.
%  ------------------------------------------------------------------
%  Estimate various norms and test for convergence.



 while ~done && (iter < maxit)
         iter   = iter  +  1;
         betal    = beta;
         beta     = betan;  
         v        = r3 * (1/beta); 
         r3       = symmlqxxxA( A,  v, varargin );% p_k
         if shift ~= 0
            r3    = r3 - shift*v;
         end
         if iter  > 1,
            r3    = r3 - (beta/betal) * r1;
         end
         alfa     = real( r3'*v );                         % alpha_k
         r3       = r3 - (alfa/beta ) * r2; % v_{k+1}
         r1       = r2;                            % v_{k-1}
         r2       = r3; 
         if ~precon,
            betan  = norm(r3);                      % beta_{k+1}
         else
            r3     = symmlqxxxM( M, r2, varargin );
            betan  = r2' * r3;
            if betan < 0
               error('The preconditioner "M" appears to be indefinite');
            end
            if betan > 0,
               betan = sqrt( betan );     
            end
         end
        pnorml  = pnorm;
        if iter == 1        
           pnorm  = norm([alfa, betan]);
        else
           pnorm  = norm([beta, alfa, betan]);
        end

%       if beta < 0, flag = 7;  break;  end
%       beta   = sqrt( beta );
%       if iter  == 1
%          Tnorm = norm([alfa, betan]);
%       else
%          Tnorm = norm([Tnorm; beta; alfa; betan]); % Frobenius norm of T_{k,k+1}
%       end  
      Anorml = Anorm;
      Anorm  = max(Anorm,pnorm);
      epsa   = Anorm * eps;
      
      if debug
         fprintf('\n  v_%d     = ', iter );  
         gprintf(sprintf('%13.3e ', v(1:min(n,5)) ));
         gprintf(sprintf('\n  alpha_%d = %13.3e, beta_%d = %13.3e, beta_%d = %13.3e ',...
                    iter, alfa, iter, beta, iter+1, betan ));
      end
    
%     Apply previous rotation Pk-1 to get           
      delta  = cs * dbarn  +  sn * alfa;
      gbar   = sn * dbarn  -  cs * alfa;

      epsln  = epslnn;
      epslnn = sn * betan;
      dbarn  =            -  cs * betan;

      if debug
         fprintf('\n  c_%d     = %13.3e, s_%d    = %13.3e ', ...
                      iter-1, cs, iter-1, sn );
         fprintf('\n  delta_%d = %13.3e, gbar_%d = %13.3e ', ...
                      iter, delta, iter, gbar );
         fprintf('\n  epsln_%d = %13.3e, dbar_%d = %13.3e ', ...
                      iter+1, epslnn, iter+1, dbarn );
      end                     
      
      if  ~disable
          if relrnorml < epsr, 
             flag = 4; 
             iter = iter - 2; 
             rnorm = rnorml; 
                break;
          end
          if relrnorml < rtol, 
             flag = 3; 
             iter = iter - 2; 
             rnorm = rnorml;     
             break;
          end
      end
      
%     Compute the next plane rotation for P.
      csl    = cs;
      snl    = sn;
      gamma  = norm([ gbar  betan ] );   
      if gamma < epsa
         flag = 6;
         if ~disable 
            break;
         end
      end
      cs     = gbar / gamma;
      sn     = betan / gamma;
      if iter == 1
         gmax = gamma; gmin = gamma;
      else
         gmax   = max( gmax, gamma );
         gmin   = min( gmin, gamma );
      end
      if debug
         fprintf('\n  c_%d     = %13.3e, s_%d    = %13.3e ', iter, cs, iter, sn );
         fprintf('\n  gamma_%d = %13.3e               ', iter, gamma );
      end
     
      Acondl = Acond;
      Acond  = gmax/gmin;
%      Anorml = Anorm;
%      Anorm  = Tnorm;
%      ynorm  = sqrt( ynorm2 );
%     epsa   = Anorm * eps;
      epsx   = Anorm * xnorm * eps;
      epsr   = Anorm * xnorm * rtol;

      if iter > 1
         varphi      = varphin  -  delta * z;
         varphin     =          -  epslnn * z;
         rnorml      = norm([varphi  varphin]);
         relrnorml   = rnorml / (Anorm*xnorml + beta1); % ||r||/(||A||||x||+||b||
         if debug & (iter > 1)
            fprintf('\n  varphi_%d = %13.3e, varphi_%d = %13.3e  ', iter, varphi, iter+1, varphin );
            fprintf('\n  rnorm_%d  = %13.3e', iter-1, rnorml );
         end
      end
      
      diag   = gbar;
      if diag == 0, diag = epsa; end
      snprod = snprod * sn;
      qrnorml = qrnorm;
      qrnorm = snprod * beta1;  % MINRES-norm(r_k)
      cgnorml = cgnorm;
      cgnorm = qrnorm * beta / abs( diag ); % CG-norm(r_k)

%     Estimate  Cond(A).
%     In this version we look at the diagonals of  L  in the
%     factorization of the tridiagonal matrix,  T = L*Q.
%     Sometimes, T(k) can be misleadingly ill-conditioned when
%     T(k+1) is not, so we must be careful not to overestimate Acond.
 %     Acondl      = Acond;
 %     if lqnorm < cgnorm
 %        Acond  = gmax / gmin;
 %     else
 %        denom  = min( gmin, abs( diag ) );
 %        Acond  = gmax / denom;
 %     end

 %    zbar   = varphi / diag;
%      z      = (snprod * zbar + bstep) / beta1;
%       x1lq   = x(1) + b1 * bstep / beta1;
%       x1cg   = x(1) + w(1) * zbar  +  b1 * z;

%     if ~show                ,   prnt = 0; end
      
%     Obtain the current Lanczos vector  v = (1 / beta)*y
%     and set up  y  for the next iteration.

%     if flag ~= 0, break; end
      
%     Update  X.
      x1last      = x(1);
      xnorml      = xnorm;
      zl2     = zl;
      zl      = z;
      if iter == 1
         z = beta1 / gamma; 
      else
         z  = -( epsln * zl2 + delta * zl ) / gamma;
      end
      if iter == 1, u = v; end
      w       = cs * u + (sn / betan) * r3;
      u       = sn * u - (cs / betan) * r3; % for next iteration
      xnorm   = norm([z  xnorm]);
      if ~disable && (xnorm >= maxxnorm) %max(d_norm, xnorm) >= maxxnorm
         flag = 6;
      else
         x       = x + z * w;
      end

      if debug
         fprintf('\n  w_%d     = ', iter );  fprintf('%13.3e ', w(1:min(n,5)) );
         fprintf('\n  x_%d     = ', iter ); fprintf('%13.3e ', x(1:min(n,5)) );
         fprintf('\n  z_%d     = %13.3e,  xnorm = %13.3e ', iter, z, xnorm);
         fprintf('\n  Tnorm_%d  = ', iter );  fprintf('%13.3e ', Tnorm );
         fprintf('\n  rnorm_%d  = ', iter -1 );  fprintf('%13.3e ', rnorml );
         fprintf('\n  Acond%d   = ', iter );  fprintf('%13.3e ', Acond );
         fprintf('\n\n');
      end
      
%     See if any of the stopping criteria are satisfied.
%     In rare cases, flag is already -1 from above (Abar = const * I).

      if flag == 0
         if iter   >= maxit  , flag = 8; end
         if Acond  >= 0.1/eps, flag = 7; end
         if epsx   >= beta1  , flag = 3; end
          
%          if cgnorm <= epsx   , flag = 2; end
%          if cgnorm <= epsr   , flag = 1; end
      end
%     ==================================================================
%     See if it is time to print something.

      prnt       = false;
      if n      <= 40         ,   prnt = true; end
      if iter    <= 10        ,   prnt = true; end
      if iter    >= maxit - 10,   prnt = true; end
      if rem(iter-1,10) == 0  ,   prnt = true; end
%      if cgnorm <= 10.0*epsx  ,   prnt = true; end
%      if cgnorm <= 10.0*epsr  ,   prnt = true; end
      if Acond  >= 0.01/eps   ,   flag = 7;    prnt = true; end
      if flag  ~= 0           ,   prnt = true; done = true; end
      if disable 
         done = 0;
         %Axnorm = norm(A*x);
         if beta1 < rtol * Anorm * xnorm 
            flag = 12; 
            done = 1; 
         end
         if iter >= maxit
             done = 1;
             flag = 8;
         end
      end
%     Accumulate the step along the direction  b, and go round again.

%       bstep  = snprod * cs * z  +  bstep;

%       gmax   = max( gmax, gamma );
%       gmin   = min( gmin, gamma );
%      ynorm2 = z^2  +  ynorm2;
     
      if fignum || (nargout > 7)
         xnormvec(iter)    = xnorm;
         if iter > 1
            resvec(iter-1)      = rnorml;
            allvecs(iter-1)     = xnorml * relrnorml ;%* relArnorml;
         end
         %if iter > 1, Aresvec(iter-1) = Arnorml; end
      end
      
      if show && prnt && flag~= 6,
         str1 = sprintf( '\n%6g %10.2e %10.2e %10.2e %10.2e %10.2e %10.2e', ...
                         iter-1, x1last, xnorml, cgnorml,  rnorml, qrnorml, relrnorml  );
         str2 = sprintf( ' %8.1e %8.1e %8.1e', Anorml, Acondl, betal);     
         %str2 = sprintf( ' %8.1e %8.1e %8.1e %8.1e',bstep/beta1 Anorml, Acondl, betal);
         gprintf(([str1  str2 ]));
         if mod(iter,10)==0, disp(' '); end
      end % show && prnt   
      
end % while
   
%  ------------------------------------------------------------------
%  End of main iteration loop.
%  ------------------------------------------------------------------
rnorm = rnorml;
%  Move to the CG point if it seems better.
%  In this version of SYMMLQ, the convergence tests involve
%  only cgnorm, so we're unlikely to stop at an LQ point,
%  EXCEPT if the iteration limit interferes.

   if (( cgnorm < rnorm ) && (cs ~= 0)) ||  disable 
      zbar   = varphi / diag;
      %bstep  = snprod * zbar  +  bstep;
      %ynorm  = sqrt( ynorm2  +  zbar^2 );
      x      = x + (z * sn / cs ) * u;
      rnorm = cgnorm;
      str5 = '\nTransfered to CG point';
   end

%  Add the step along  b.

%    bstep  = bstep / beta1;
%    y      = b;
%    if precon,  y =  symlqxxxM(M, b, varargin );  end
%    x      = bstep * y + x;

%  Compute the final residual,  r1 = b - (A - shift*I)*x.
            
%    y      = symmlqxxxA( A, x, varargin );
%    y      = (- shift) * x + y;
%    r1     = b - y;
%    rnorm  = norm ( r1 );
%   xnorm  = norm (  x );
    
if fignum || (nargout > 7)
   resvec(iter) = rnorm;
  % Aresvec(iter) = Arnorm;
end

%% print final iteration
if show && prnt & flag ~=3 & flag ~=4
   str1 = sprintf( '\n%6g %10.2e %10.2e %10.2e %10.2e %10.2e %10.2e', ...
                         iter, x1last, xnorml, cgnorml,  rnorml, relrnorml, qrnorml );
   str3 = sprintf( ' %8.1e %8.1e %8.1e'      , Anorml, Acondl, betal);
   str4 = sprintf( '\n%91.1e               ',                  beta);
   gprintf([str1 str3]);
   gprintf(str4);
   if ~isequal(str5,'') fprintf(str5);
   end
end % show && prnt

%  ==================================================================
%%  Display final status.
%  ==================================================================
if show
   disp( space )
   gprintf( [ last sprintf( ' flag    =  %3g               iter =  %5g\n',...
                           flag,                      iter  ) ] );
   gprintf( [ last sprintf( ' Anorm   =  %12.4e      Acond =  %12.4e\n' ,...
                           Anorm,                 Acond  ) ] );
   gprintf( [ last sprintf( ' rnorm   =  %12.4e      xnorm =  %12.4e\n' ,...
                           rnorm,                 xnorm  ) ] );
   gprintf( [ last msg(flag,:) ]);
   disp( space )
end

%% Plot residual norm, solution norm and norm of A r_k

if (fignum>0 || (nargout > 7)) && iter>1 
   resvec      = resvec(1:iter);
   Aresvec     = Aresvec(1:iter);
   xnormvec    = xnormvec(1:iter);

   if isbestiter
      allvecs     = allvecs(1:iter);
   end
   if fignum > 0 , figure(fignum); end
   
   subplot(2,2,1);
   semilogy(1:iter, max(resvec,realmin), '-r');  hold on;  
   tmpvar = 10^floor(log10(rnorm));
   semilogy([1,iter], [tmpvar, tmpvar], '-k');  hold off;  %black
   title('||r_k||');
   xlabel('Iterations');

   subplot(2,2,2);
   semilogy(1:iter, max(Aresvec,realmin), '-r'); axis tight;
   title('||Ar_k||');
   xlabel('Iterations');
  
   subplot(2,2,3);
   semilogy(1:iter, max(xnormvec,realmin), '-r'); axis tight; hold on;
   semilogy(1:iter, max(xnormvec,realmin), '--b'); axis tight; hold off;
   title('||x_k||');
   legend('||x_k||', '||x_{k-1}||+|\tau_k| ||w_k||','Location','SouthEast');
   xlabel('Iterations');
   
   if isbestiter
       subplot(2,2,4);
       semilogy(1:iter, max(allvecs, realmin), '-r'); axis tight;
       title(['||r_k||', '||Ar_k||', '||x_k||']);
       xlabel('Iterations');
       [dummy, bestiter] = min(allvecs(2:iter)); % xnormvec(1) = 0;
   end
end
clear symmlqxxxM
% end SYMMLQ


function p = symmlqxxxA( A,x,varargin )

%        p = symmlqxxxA( A,x,varargin )
% computes p = Ax for a matrix or operator A defined by input parameter A.

%-----------------------------------------------------------------------
% 17 Oct 2003: Default A*x function for minres.m.
%              User parameters A, varargin are passed thru to here.
%-----------------------------------------------------------------------

  if isa(A,'function_handle')
     p = A( x,varargin );
  else
     p = A*x;
  end
  
function p = symlqxxxM( M,x,varargin )

%        p = symlqxxxM( M,x,varargin )
% solves My = x for a matrix or operator M defined by input parameter M.

%-----------------------------------------------------------------------
% 17 Oct 2003: Default M\x for minres.m.
%              User parameters M, varargin are passed thru to here.
% Precondition: M symmetric positive definite.
% Note: CHOL works with sparse matrices.
%-----------------------------------------------------------------------
 
  if isa(M,'function_handle')
     p = M( x,varargin );
  else
     %p = M\x; % calls CHOL in each iteration!!!
     %The following calls CHOL only once throughout all iterations
     if ~exist('R','var')
        persistent R % clear symlqxxxM before existing MINRES-B.
        R = chol(M); 
     end 
     p = R' \ x;
     p = R  \ p;
  end

%-----------------------------------------------------------------------
% End private function symlqxxxM
%-----------------------------------------------------------------------