 function [issym,Anorm] = IsOpSym6( A, n, varargin );
% IFOPSYM5: statistically test if a large operator/matrix is symmetric
%  
% USAGE:
%         [issym,Anorm] = IsOpSym6( A, n, varargin );
%
% DESCRIPTION:
%        z = |x'(Ay) - y'(Ax)| for two random generators x and y.
%    This is a statistical test and therefore there is a chance that 
%    it will make the wrong conclusion. See testIsOpSym.m examples 25 & 26.
%
% INPUTS:
%    A   matrix or operator, dense or sparse matrix
%            or a function handle
%    n       order of matrix or operator A
%
% OUTPUTS:
%    issym boolean
%          0:  if we recommmend concluding unsymmetric
%          1:  if we recommmend concluding symmetric
%    Anorm   An estimate of norm(A,2).
%
% EXAMPLE:
%    A = [ 0 0; 1 1]; 
%    [issym, Anorm] = IsOpSym6( A, n );
%
% SEE ALSO:
%    testRandVecGenSpeed, testIsOpSym, minresSOL28
%
% REFERENCES:
%    ---
%
% MODIFICATION HISTORY:
%    10/25/2004: (1) change z from
%                        z = | b'A'Ab - b'AAb | 
%                    with fixed vector b (usually b in of Ax = b)
%                    to                   
%                        z = |x'(Ay) - y'(Ax)|, with random vectors x and y
%                (2) change thresholds epsa from 
%                        epsa  = (s  + eps) * eps^(1/3);
%                    to                   
%                        epsa = n * eps  
%                        epsa = 1.01 * n * eps 
%                     
%    11/02/2004: (1) reverse thresholds epsa from: 
%                       epsa = n * eps  
%                       epsa = 1.01 * n * eps 
%                     back to 
%                       epsa  = ( abs( s ) + eps ) * eps^(1/3);
%                     as the orginal ones do not work for even simple case 
%                     like 
%                        A    = [ 1 2; 2 1] 
%                 (2) do  
%                        w' *  x
%                     as 
%                        (w/wnorm) * (x/xnorm) *(wnorm * xnorm)
%                     to reduce chances of overflow.
%     11/05/2004: (1) do  
%                        w' *  x as 
%                     as 
%                        ( (w/wnorm)' * x ) * wnorm 
%                     as we understand ||x|| is of order 10 if 
%                     x(i) ~ i.i.d. N(0,1) for i = 1,..., n even for n as
%                     large as n = 10^8
% 31 Mar 2005: A must be a matrix or a function handle.
% 28 Apr 2005: Output Anorm to help MINRES at iter = 1.
%
% KNOWN BUGS:
%    MM/DD/2004: ---
%
% AUTHOR: Sou Cheng Choi, SCCM, Stanford University
%         Michael Saunders, SOL, Stanford University
%
% CREATION DATE: 10/25/2004

 
BIGNO = 1e200; % ARBITRARY constant!

STATE = randn('state');       % Save state
randn('state',0);             % 
x     = randn(n,1);           % norm of x is usually of order O(10)
y     = randn(n,1);           % norm of y is usually of order O(10)
randn('state',STATE);         % Restore state

xnorm = norm(x);
ynorm = norm(y);

w     = Aop( A, y, varargin );
wnorm = norm(w);
Anorm = wnorm/ynorm;

if wnorm > BIGNO
  s = ((w/wnorm)' * x) * wnorm; %  w' * x 
else
  s = w' * x;
end

w     = Aop( A, x, varargin );
wnorm = norm(w);
Anorm = max( Anorm, wnorm/xnorm );

if wnorm > BIGNO,
  t = ((w/wnorm)' * y) * wnorm; % w' * y;
else
  t = y' * w;
end

z     =  abs(s - conj(t));
epsa  = (abs(s) + abs(t) + eps) * eps^(1/3); % threshold, default epsa
issym = (z < epsa);


function y = Aop( A, x, varargin )

%        y = Aop( A, x, varargin )
%    computes y = Ax for a matrix A defined by input parameter A.
% 
%    matrix or operator, dense or sparse matrix (preferably sparse!)
%            or the name of a function such that
%               x = A( y, n, iw, rw )
%            returns the product x = Ay for any given vector y
%    n       order of matrix or operator A
%    y       vector of length n
%    iw      extra input argument to function A
%    rw      extra input argument to function A
%-----------------------------------------------------------------------
% 17 Oct 2003: Default A*x function for minres.m.
%              User parameters A, n, iw, rw are passed thru to here.
% 31 Mar 2005: A must be a matrix or a function handle.
%-----------------------------------------------------------------------
  if isa(A,'function_handle')
    y = A( x, varargin );
  else
    y = A*x;
  end

%-----------------------------------------------------------------------
% End private function Aop
%-----------------------------------------------------------------------
