function [c, s, d] = SymGivens2(a, b)
 
% SYMGIVENS2: Stable Symmetric Givens rotation plus reflection 
%  
%  USAGE:
%     [c, s, d] = SymGivens2(a, b)
%
%  INPUTS:
%    a      first element of a two-vector  [a; b]      
%    b      second element of a two-vector [a; b] 
%
%  OUTPUTS:
%    c      cosine(theta), where theta is the implicit angle of rotation
%           (counter-clockwise) in a plane-rotation
%    s      sine(theta)
%    d      two-norm of [a; b]
%  
%  DESCRIPTION:
%     Stable symmetric Givens rotation that gives c and s such that 
%        [ c  s ][a] = [d],
%        [ s -c ][b]   [0]  
%     where d = two norm of vector [a, b],
%        c = a / sqrt(a^2 + b^2) = a / d, 
%        s = b / sqrt(a^2 + b^2) = b / d.
%     The implementation guards against overlow in computing sqrt(a^2 + b^2).
%
%  EXAMPLE:
%      description
%
%  SEE ALSO:
%     TESTSYMGIVENS.m, 
%     PLANEROT (MATLAB's function) --- 4 divisions while 2 would be enough, 
%     though not too time-consuming on modern machines
%  
%  REFERENCES:
%    Algorithm 4.9, stable *unsymmetric* Givens rotations in
%     Golub and van Loan's book Matrix Computations, 3rd edition. 
%
%  MODIFICATION HISTORY:
%    10/06/2004: replace d = norm([a,b]) by 
%                        d = a/c if |b| < |a| or b/s otherwise.
%    10/07/2004: first two cases (either b or a == 0) rewritten to make sure 
%                (1) d >= 0 
%                (2) if [a,b] = 0, then c = 1 and s = 0 but not c = s = 0. 
%
%  KNOWN BUGS:
%     MM/DD/2004: description
%
%  AUTHOR: Sou-Cheng Choi, SCCM, Stanford University
%          Michael Saunders, SOL, Stanford University
%
%  CREATION DATE: 09/28/2004
%

if b == 0,
  if a == 0, 
    c = 1; 
  else 
    c = sign(a);  % NOTE: sign(0) = 0 in MATLAB
  end 
  s = 0;
  d = abs(a);

elseif a == 0,
  c = 0;
  s = sign(b);
  d = abs(b);
  
elseif abs(b) > abs(a)
  t = a / b;
  s = sign(b) / sqrt(1 + t^2); 
  c = s*t;
  d = b / s; % computationally better than d = a / c since |c| <= |s|
  
else 
  t = b / a; 
  c = sign(a) / sqrt(1 + t^2); 
  s = c*t;
  d = a / c; % computationally better than d = b / s since |s| <= |c|
end

 

 
