function test_minresqlp_fig3(egNo)
%
% FILE_NAME: test_minresqlp_fig3.m
%  
% USAGE:
%     test_minresqlp_fig3(egNo);
%
% INPUTS:
%    egNo  integer, 1 or 2, represents an example
%  
% OUTPUTS:
%    --     
%
% EXAMPLE:
%     test_minresqlp_fig3(2);
%
% SEE ALSO:
%    testminresQLP27
%
% REFERENCES:
%    (1) S.C. Choi, Iterative Methods for Singular Linear Equations and 
%    Least-Squares Problems. PhD thesis, Stanford University, December 2006.
%    
%    (2) C. C. Paige and M. A. Saunders, Solution of sparse
%    indefinite systems of linear equations, SIAM J. Numerical Analysis 12,
%    617-629  (1975). 
%
% MODIFICATION HISTORY:
%    08/23/2006: Renamed testminresQLP27.m to test_minresqlp_fig3.m.
%                Cleaned up the script. 
%                Updated minresQLP43 to minresQLP49. 
%                Updated minresQLPs7 to minresQLP. 
%
% AUTHORS: Sou-Cheng (Terrya) Choi, ICME, Stanford University
%          Michael Saunders, SOL / ICME, Stanford University
%
% CREATION DATE: 08/23/2009
%
close all;
format compact
format short e

disp('======================= Set up input parameters ========================');

rand('state', 0);  

M        = [];
iw       = [];
rw       = [];
Acondlim = [];
precond  = false;
shift    = 0;
show     = false; %true;
maxxnorm = 1e9;
rtol     = 100*eps;
debug    = 0;
isbestiter = 0;

errevd   = -1;  
errtevd  = -1;   
errMM    = -1;
err      = -1;
err1     = -1;
err2     = -1;
errpcg   = -1;
errsym   = -1;
errMS    = -1;
errsq    = -1;
mu       = 1;
isdiag   = 0;
disable = 0;
TranCond = 1e7;
restart = [];
switch egNo 
   case 0
       rand('state',0);
       A = loadharvard500; 
       n = length(A);
       id = -5;
       title = 'Cleve Moler''s harvard500.mat';
       b = rand(n,1);
       itnlim   = n;
       show     = true;
       shift    = 0;
       maxxnorm = 1e2;
       shift0   = 4;%-2;
       shiftn   = 4.001; %5.5;
       shifth   = 0;
       mu       = 10;
       rtol     = 1e-4;
   case 1 % small example n = 25  % almost compatible
       %%%%% Good Example !!!! %%%%%%
       load('Davis1177.mat');
       A        = Problem.A;
       n        = length(A);
       id       = Problem.id;
       title    = Problem.title;
       itnlim   = n;
       show     = true;
       shift    = 0;
       maxxnorm = 1e2;
       shift0   = 4;%-2;
       shiftn   = 4.001; %5.5;
       shifth   = 0;
       mu       = 10;
       b        = (A-shift*speye(n))*rand(n,1)  ; %  compatible 
       rtol     = 1e-11;
  case 2 % small example n = 25

       load('Davis1177.mat');
       A        = Problem.A;
       n        = length(A);
       id       = Problem.id;
       title    = Problem.title;
       itnlim   = n*2;
       show     = true;
       smallr   = 1e-5*rand(n,1);
       b        = (A-shift*speye(n))*rand(n,1) + smallr; % almost compatible
       rtol     = 1e-14;
       shift    = 0;% eigenvalue
       shift0   = 4;%-2;
       shiftn   = 4.001; %5.5;
       shifth   = 0;
       maxxnorm  = 1e4; %1e7
  case 3 % small example n = 25

       load('Davis1177.mat');
       A        = Problem.A;
       n        = length(A);
       id       = Problem.id;
       title    = Problem.title;
       itnlim   = n*2;
       show     = true;
       b        = rand(n,1); % incompatible
       rtol     = 1e-11;
       shift    = 0;% eigenvalue
       shift0   = 4;%-2;
       shiftn   = 4.001; %5.5;
       shifth   = 0;
       maxxnorm = 1e4;
       
  case 4 % small example n = 25

       load('Davis1177.mat');
       A        = Problem.A;
       [A, D]   = diagscal7(A);
       isdiag   = 1;
       n        = length(A);
       id       = Problem.id;
       title    = Problem.title;
       itnlim   = n*2;
       show     = true;
       b        = rand(n,1); % incompatible
       rtol     = 1e-10;
       shift    = 0;% eigenvalue
       shift0   = 4;%-2;
       shiftn   = 4.001; %5.5;
       shifth   = 0;
       maxxnorm = 1e5;
   
end

beta1 = norm(b);

 
filename = ['Davis', num2str(id), 'Case', num2str(egNo), 'Eig'];
diary([filename,'.txt'])

fprintf('\nComputing EVD and TEVD of A and the solutions...')

tic
[tx, trankA, tcondA, ts, xevd, rankA, condA, normA, ...
 errVD, terrVD, ortherrVD, tortherrVD, s, V, tsInd, snzInd] = tevd7(A,b,[],[],[],mu);
toctx = toc
fprintf('\nSaving EVD and TEVD of A in %s...', filename)
save(filename, 's', 'V');
ss = s;
if shift ~= 0 
   ss = ss - shift;
end


xevdnorm = norm(xevd);
xsnorm   = norm(tx);
if shift ~= 0
   revd      = A*xevd - shift*xevd - b;
   rs        = A*tx - shift*tx - b;
   Arevdnorm = norm(A*revd-shift*revd);
   Arsnorm   = norm(A*rs-shift*rs);
else
   revd      = A*xevd - b;
   rs        = A*tx - b;
   Arevdnorm = norm(revd'*A);
   Arsnorm   = norm(rs'*A);
end  
revdnorm = norm(revd);
rsnorm = norm(rs);
if exist('xevd','var'); 
   errtevd  = norm(tx - tx);
   errevd   = norm(tx - xevd);
   normA    = max(ts);
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Print Inputs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

nnzA = nnz(A-shift*speye(n));
fprintf('\nMatrix ID = %10d,   title    = %s', id, title);
fprintf('\nn         = %10d,   maxit    = %10d,   nnz       = %10d,  nnz / n ~= %6d.',n, itnlim,nnzA,ceil( nnzA / n));
fprintf('\nshift     = %10.3e,   tol      = %10.0e,  maxxnorm   = %10.0e,  maxcond = %8.1e.', shift, rtol, maxxnorm, Acondlim);
fprintf('\n||b||     = %8.1e.',  beta1);

if shifth > 0
   fprintf('Finding a shift that makes A ill-conditioned...');
   [shiftnew, trankA, tcondA, rankA, condA] = illCondShift1(A, shift0, shiftn, shifth)
   if ~isempty(shiftnew) 
      shift = shiftnew;
   end
   %keyboard;
end

if ~isequal(A,A') 
   warning('Symmetrizing A by (A+A'')/2...');
   A = (A+A')/2;
end
       
 

%
%% Iterative methods
%
disp('========================= SYMMMLQ SOL=====================');
check = 1;
fignum = 0; 
 
tic
[ xsym1, istopSYM1, itnsym1, rnormsym1, Anormsym1, Acondsym1, xnormsym1, resvecsym1, Aresvecsym1, xnormvecsym1 ] =... 
       symmlqSOL( A, b, rtol, itnlim, maxxnorm, M, shift, show, fignum, debug );
tocsym1 = toc
rsym1         = b-A*xsym1+shift*xsym1;
rnormsymout1  = norm(rsym1);
Arnormsymout1 = norm(A*rsym1 - shift*rsym1); 
if exist('xevd','var')
   errsym1 = norm(tx - xsym1);
end

disp('========================= SYMMMLQ SOL3=====================');
check = 1;
fignum = 0;% 3;
 
tic
[ xsym, istopSYM, itnsym, rnormsym, Anormsym, Acondsym, xnormsym, resvecsym, Aresvecsym, xnormvecsym ] =... 
       symmlqSOL3( A, b, rtol, itnlim, maxxnorm, M, shift, show, fignum, debug, disable );
tocsym = toc
rsym         = b-A*xsym+shift*xsym;
rnormsymout  = norm(rsym);
Arnormsymout = norm(A*rsym - shift*rsym); 
if exist('xevd','var')
   errsym = norm(tx - xsym);
end

disp('========================= Old SOL MINRES =====================');

tic
[ x, istop, itn, rnorm, Arnorm, Anorm, Acond, xnorm ] =... 
     minresSOL(n, b, A, M, iw, rw, ...
          precond, shift, show, check, itnlim, rtol);
tocm = toc 
r         = b-A*x+shift*x;
rnormout  = norm(r);
Arnormout = norm(A*r - shift*r); 
if exist('xevd','var'), 
   err = norm(tx - x); 
end

disp('========================= minresSOL69 ========================');
fignum = 0;  
testsym = 1;
tic
 
[x1, istopSOL, itn1, rnorm1, Arnorm1, Anorm1, Acond1, xnorm1, Axnorm, ...
           resvec, Aresvec, xnormvec, pnormvecSOL, gamavecSOL, alfavecSOL, betavecSOL, VV,...
           resvecdirect, relresvec, Aresvecdirect, relAresvec,  pnormdirectvec, Axnormvec ] = ...
           minresSOL69( A, b, rtol, itnlim, M, shift, maxxnorm, Acondlim,  show, testsym, ...
           fignum, debug, disable, isbestiter);
 
tocSOL = toc
r1         = b-A*x1+shift*x1;
rnorm1out  = norm(r1);
Arnorm1out = norm(A*r1 - shift*r1);
if exist('xevd','var'), 
   err1 = norm(tx - x1); 
end
        
disp('========================= minresQLP49 ========================');

 
fignum     = 0;% 3;
 
tic
 
   [x2, istopQLP, itn2, rnorm2, Arnorm2, Anorm2, Acond2, xnorm2, Axnorm2,  ...
     resvec2, Aresvec2, xnormvec2, pnormvecQLP, gamavecQLP, alfavecQLP, betavecQLP, veplnvecQLP, etavecQLP, VVmatQLP, ...
     resvecdirectQLP, relresvecQLP, AresvecdirectQLP, relAresvecQLP,xnormdirectvecQLP, pnormdirectvecQLP, Axnormvec2QLP  ] = ...
     minresQLP49( A,  b,  rtol, itnlim,  M, shift, maxxnorm, Acondlim, show, testsym, fignum,...
             debug,disable, isbestiter);
 
tocQLP = toc;   
r2         = b-A*x2+shift*x2;
rnorm2out  = norm(r2);
Arnorm2out = norm(A*r2 - shift*r2);
if exist('xevd','var'), 
   err2 = norm(tx - x2); 
end

 
[x2s, istopQLPs, itn2s, Mitn2s, QLPitn2s, rnorm2s, Arnorm2s, Anorm2s, Acond2s, xnorm2s ] = ...
    minresQLP( A,  b,  rtol, itnlim,  M, shift, maxxnorm, Acondlim, TranCond, show);
 
pwdpath   = pwd;
curMINRES = which('minresSOL69');
curpath   = curMINRES(1:end-14);
%rmpath(curpath);
currentMINRES = which('minres','-all');
for i=1:length(currentMINRES)
    if findstr(char(currentMINRES(i)),'sparfun'), ind=i; end
end
Matlabpath = char(currentMINRES(ind));
cd(Matlabpath(1:end-9));
cd private;

disp('');
disp('========================= Matlab PCG =====================');
tic
 
[xpcg,flagpcg,rnormpcg,itnpcg,resvecpcg,xnormvecpcg] = pcgMatlab(A-shift*speye(n),b,rtol,itnlim);
 
tocpcg = toc;
resvecpcg = resvecpcg(2:end); % resvecpcg(1) = beta1 = ||r0||
%itnpcg = length(resvecpcg); 
xnormvecpcg = xnormvecpcg(1:itnpcg);
rpcg         = b-A*xpcg+shift*xpcg;
rnormpcgout  = norm(rpcg);
Arnormpcgout = norm(A*rpcg - shift*rpcg); 
if exist('xevd','var')
   errpcg = norm(tx - xpcg);
end

 

disp('===================== Matlab MINRES / SYMMLQ =========================='); 
tic
[ xMM , flagmin, rnormMM , itnMM, resvecMM, resveccgMM ] = ...
    minres( A-shift*speye(n), b, rtol, itnlim);
tocMM = toc;
resvecMM = resvecMM(2:end); % resvecMM(1) = ||r0|| = beta1
%itnMM = length(resvecMM);
rMM         = b-A*xMM+shift*xMM;
rnormMMout  = norm(rMM);
ArnormMMout = norm(A*rMM - shift*rMM);

tic
[ xMS, flagsym, rnormMS , itnMS, resvecMS, resveccgMS ] = ...
    symmlq( A-shift*speye(n), b, rtol, itnlim);
tocMS = toc;
resvecMS = resvecMS(2:end); % resvecMS(1) = ||r0|| = beta1
%itnMS = length(resvecMS);
rMS         = b-A*xMS+shift*xMS;
rnormMSout  = norm(rMS);
ArnormMSout = norm(A*rMS - shift*rMS);


disp('===================== Matlab GMRES ==========================');
if isempty(restart)
   itnlimg = itnlim;
else
   itnlimg = itnlim/restart;
end
tic
[xg,flagg,relresg,iterg0,resvecg] = gmres(A-shift*speye(n),b,restart,rtol,itnlimg,M);
tocg = toc;
resvecg = resvecg(2:end); % resvecg(1) = ||r0|| = beta1
iterg = iterg0(1)*iterg0(2);
rg         = b-A*xg+shift*xg;
rnormgout  = norm(rg);
Arnormgout = norm(A*rg - shift*rg);
     
disp('===================== Matlab BiCG ==========================');       
tic
[xb,flagb,relresb,iterb,resvecb] = bicg(A-shift*speye(n),b,rtol,itnlim,M);
tocb = toc;
resvecb = resvecb(2:end); % resvecb(1) = ||r0|| = beta1
%iterb = length(resvecb);
rb         = b-A*xb+shift*xb;
rnormbout  = norm(rb);
Arnormbout = norm(A*rb - shift*rb);

disp('===================== Matlab BiCGStab ==========================');
tic
[xbc,flagbc,relresbc,iterbc,resvecbc] = bicgstab(A-shift*speye(n),b,rtol,itnlim,M);
tocbc = toc;
resvecbc = resvecbc(2:end); % resvecbc(1) = ||r0|| = beta1
%iterbc = length(resvecbc);
rbc         = b-A*xbc+shift*xbc;
rnormbcout  = norm(rbc);
Arnormbcout = norm(A*rbc - shift*rbc);

disp('===================== Matlab QMR ==========================');
tic
[xq,flagq,relresq,iterq,resvecq] = qmr(A-shift*speye(n),b,rtol,itnlim,M);
tocq =  toc;
resvecq = resvecq(2:end); % resvecq(1) = ||r0|| = beta1
%iterq = length(resvecq);
rq         = b-A*xq+shift*xq;
rnormqout  = norm(rq);
Arnormqout = norm(A*rq - shift*rq);


disp('===================== Matlab LSQR ==========================');
tic
[xlm,flaglm,relreslm,iterlm,resveclm] = lsqr(A-shift*speye(n),b,rtol,itnlim,M);
toclm =  toc;
resveclm = resveclm(2:end); % resveclm(1) = ||r0|| = beta1
%iterlm = length(resveclm);
rlm         = b-A*xlm+shift*xlm;
rnormlmout  = norm(rlm);
Arnormlmout = norm(A*rlm - shift*rlm);


cd(pwdpath);
addpath(curpath);
curMINRES = which('minresSOL69');

disp('===================== SQMR ==========================');
tic
[xsq,flagsq,relressq,itersq,resvecsq] = SQMR1(A-shift*speye(n),b,rtol,itnlim,M);
tocsq =  toc;
%itersq = length(resvecsq);
rsq         = b-A*xsq+shift*xsq;
rnormsqout  = norm(rsq);
Arnormsqout = norm(A*rsq - shift*rsq);
disp('===================== LSQR ==========================');
damp = 0;
atol = rtol;
btol = rtol;
tic
[ xl, flagl, iterl, rnorml, r2norml, anorml, acondl, arnorml, xnorml, axnorml,...
      resvecl, Aresvecl, xnormvecl, Axnormvecl] = ...
lsqrSOL( n, n, A, [], [], b, damp, atol, btol, Acondlim, itnlim, maxxnorm, show, disable );
tocl = toc;
rl         = b-A*xl+shift*xl;
rnormlout  = norm(rl);
Arnormlout = norm(A*rl - shift*rl);

if exist('xevd','var'), 
    errMM = norm(tx - xMM);
    errMS = norm(tx - xMS);
    errg  = norm(tx - xg);
    errb  = norm(tx - xb);
    errbc = norm(tx - xbc);
    errq  = norm(tx - xq);
    errsq  = norm(tx - xsq);
    errlm = norm(tx - xlm);
    errl  = norm(tx - xl);
end

if egNo == 1
    figure(2)
    mfig = 1; nfig = 3;
    h = subplot(mfig,nfig,1); 
    semilogy(resvecdirect,'b-');
    hold on;  
    semilogy(resvecdirectQLP,'r-.');
    set(h, 'YLim',  [1e-12 1]);
    set(h, 'YTick', [1e-12 1e-6 1]);
    set(h, 'XLim',  [1 10]);
    set(h, 'XTick', 0:2:10);

    options={'Interpreter','latex'};
    hl = legend('MINRES', 'MINRES-QLP',3);
    %set(hl, options{:});
    set(hl,'fontsize',8);
    set(hl,'position',[1.3591e-001  4.1373e-001  1.6667e-001  6.5000e-002]);
    axis square

    h = subplot(mfig,nfig,2);
    semilogy(1:itn1, Aresvec, 'b-', 'LineWidth',1 );  
    hold on; axis square; 
    set(h, 'YLim',  [1e-12 1]);
    set(h, 'YTick', [1e-12 1e-6 1]);
    set(h, 'XLim',  [1 10]);
    set(h, 'XTick', 0:2:10);

    semilogy(1:itn2, Aresvec2, 'r-.', 'LineWidth',1);  
    hold on; axis square; 

    h = subplot(mfig,nfig,3);
     
    plot(1:itn1, xnormvec, 'b-', 'LineWidth',1 );
    hold on; axis square; 

    plot(1:itn2, xnormvec2, 'r-.', 'LineWidth',1);  
    hold on; axis square; 
     
    set(h, 'YLim',  [2 4]);
    set(h, 'YTick', 2:4);
    set(h, 'XLim',  [1 10]);
    set(h, 'XTick', 0:2:10);

    subplot(mfig,nfig,1);
    ylabel('$ \| r_k \|_2 $',options{:});
    xlabel('$k$',options{:})
    subplot(1,3,2);
    ylabel('$ \| A r_k \|_2 $',options{:});
    xlabel('$k$',options{:})
    subplot(1,3,3);
    ylabel('$ \| x_k \|_2 $',options{:});
    xlabel('$k$',options{:}) 
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Postprocessing of solutions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Print Inputs again
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
nnzA = nnz(A-shift*speye(n));

if id > 0
   fprintf('\nMatrix ID = %4d, title  = %s', id, title);
else
   fprintf('\nTitle  = %s', title);
end
if isempty(Acondlim)
    Acondlim = 1e14;
end
    fprintf('\nn     = %5d,   maxit = %5d,   nnz    = %5d,   nnz / n = %5d.',...
               n,            itnlim,          nnzA,           ceil( nnzA / n));
    gprintf(sprintf('\nshift = %5.2e,  tol = %5.0e, maxxnorm = %5.0e,  maxcond = %5.0e, ||b|| = %5.1e. ',...
               shift,          rtol,          maxxnorm,           Acondlim,               beta1));  

    fprintf('\n===================================== Results ======================================');

    if exist('xevd','var'), 
       fprintf('\nNumerical rank = %d.',  trankA);
       maxs       = max(abs([ss(1), ss(end)])); % ||A||_2
       zerotol    = mu * maxs * eps;
       poseigind  = find(ss      >  zerotol);
       negeigind  = find(ss      < -zerotol);
       zeroeigind = find(abs(ss) <= zerotol);
       posno      = length(poseigind); % ts > ||A||_2 * eps
       negno      = length(negeigind);
       zerono     = length(zeroeigind);
       fprintf('\nNo. of positive eigenvalue(s)    = %4d', posno); 
       if posno > 0  
          gprintf(sprintf(': between %10.2e and %10.2e.', min(ss(poseigind)) , max(ss(poseigind))) ); 
       end
       fprintf('\nNo. of almost zero eigenvalue(s) = %4d', zerono); 
       if zerono > 0 
          gprintf(sprintf(': between %10.2e and %10.2e.', min(ss(zeroeigind)), max(ss(zeroeigind)) )); 
       end
       fprintf('\nNo. of negative eigenvalue(s)    = %4d', negno);  
       if negno > 0
          gprintf(sprintf(': between %10.2e and %10.2e.',  min(ss(negeigind)) , max(ss(negeigind)) )); 
       end
    end
    fprintf('\n');

    % header
    fprintf('\nMethods:');
  
    if exist('xevd','var')   
      fprintf('\n 1 : EVD.                ');         gprintf(sprintf('|| AV - VD ||_2 / ||A||_2 = %10.2e.', errVD/normA));       
      fprintf('\n                         ');         gprintf(sprintf('|| V''V - I ||_2 = %10.2e', ortherrVD));   
      fprintf('\n 2 : TEVD.               ');         gprintf(sprintf('|| AV - VD ||_2 / ||A||_2 = %10.2e.', terrVD/normA));   
      fprintf('\n                         ');         gprintf(sprintf('|| V''V - I ||_2 = %10.2e', tortherrVD) );        
    end
    fprintf('\n 3 : Matlab 7 PCG.       ');         methodno = 3; printflagmsg(methodno, flagpcg);
    fprintf('\n 4 : Matlab 7 SYMMLQ.    ');         methodno = 4; printflagmsg(methodno, flagsym);
    fprintf('\n 5 : SYMMLQ SOL.         ');         methodno = 5; printflagmsg(methodno, istopSYM1);
    fprintf('\n 6 : SYMMLQ SOL3.        ');         methodno = 6; printflagmsg(methodno, istopSYM);
    fprintf('\n 7 : Matlab 7 MINRES.    ');         methodno = 7; printflagmsg(methodno, flagmin);
    fprintf('\n 8 : MINRES SOL.         ');         methodno = 8; printflagmsg(methodno, istop);
    fprintf('\n 9 : MINRES SOL69.       ');         methodno = 9; printflagmsg(methodno, istopSOL);
    fprintf('\n 10: MINRES QLP49.       ');         methodno = 10; printflagmsg(methodno, istopQLP); 
                                                    fprintf(' trancond = %5.0e, Mitn = %d, Qitn = %d.', TranCond, Mitn2s, itn2 - Mitn2s); 

    fprintf('\n 11: Matlab 7 LSQR.      ');         methodno = 11; printflagmsg(methodno, flaglm);
    fprintf('\n 12: LSQR SOL.           ');         methodno = 12; printflagmsg(methodno, flagl);

    fprintf(['\n 13: Matlab 7 GMRES(', num2str(restart), '). ']);         
                                                    methodno = 13; printflagmsg(methodno, flagg);
    fprintf('\n 14: SQMR.               ');         methodno = 14; printflagmsg(methodno, flagsq);
    fprintf('\n 15: Matlab 7 QMR.       ');         methodno = 15; printflagmsg(methodno, flagq);
    fprintf('\n 16: Matlab 7 BICG.      ');         methodno = 16; printflagmsg(methodno, flagb);
    fprintf('\n 17: Matlab 7 BICGSTAB.  ');         methodno = 17; printflagmsg(methodno, flagbc);
  
   
    
    fprintf('\n');
    
    fprintf(['\nMethod A*v  x(1)       ||x||               ||e||      ',...
             '||r||               ||Ar||              ||A||      K(A)     time']); 
    fprintf(['\n                       direct   recurrent =||x-xTEVD|| direct   ',...
             'recurrent  direct  recurrent       \n']); 
    emptystr = '   --    ';
    emptstr2 = '    --';
   
    if exist('xevd','var')
       gprintf(sprintf('\n%s %s %11.3e %11.3e %s %10.1e %10.1e %s %10.1e %s %10.1e %10.1e %10.1e',...
        '1  ', emptstr2, xevd(1), xevdnorm, emptystr, errevd, revdnorm,  emptystr,...
        Arevdnorm, emptystr, normA, condA, toctx));
       gprintf(sprintf('\n%s %s %11.3e %11.3e %s %10.1e %10.1e %s %10.1e %s %10.1e %10.1e %10.1e',...
        '2  ', emptstr2, tx(1), xsnorm, emptystr, errtevd, rsnorm,  emptystr, ...
        Arsnorm, emptystr, normA, tcondA, toctx));
    end
   
    gprintf(sprintf('\n%s %6d %11.3e %11.3e %s %10.1e %10.1e %10.1e %10.1e %s %s %s %10.1e',...
     '3  ', itnpcg, xpcg(1), norm(xpcg), emptystr, errpcg, rnormpcgout,  rnormpcg*beta1, ...
     Arnormpcgout, emptystr, emptystr, emptystr, tocpcg));
    gprintf(sprintf('\n%s %6d %11.3e %11.3e %s %10.1e %10.1e %10.1e %10.1e %s %s %s %10.1e',...
     '4  ', itnMS, xMS(1), norm(xMS), emptystr, errMS, rnormMSout, rnormMS*beta1,...
      ArnormMSout, emptystr, emptystr, emptystr, tocMS));
    gprintf(sprintf('\n%s %6d %11.3e %11.3e %10.1e %10.1e %10.1e %10.1e %10.1e %s %10.1e %10.1e %10.1e',...
     '5  ', itnsym1, xsym1(1), norm(xsym1), xnormsym1, errsym1, rnormsymout1,  rnormsym1, ...
     Arnormsymout1, emptystr, Anormsym1, Acondsym1, tocsym1));
    gprintf(sprintf('\n%s %6d %11.3e %11.3e %10.1e %10.1e %10.1e %10.1e %10.1e %s %10.1e %10.1e %10.1e',...
     '6  ', itnsym, xsym(1), norm(xsym), xnormsym, errsym, rnormsymout,  rnormsym, ...
     Arnormsymout, emptystr, Anormsym, Acondsym, tocsym));
    gprintf(sprintf('\n%s %6d %11.3e %11.3e %s %10.1e %10.1e %10.1e %10.1e %s %s %s %10.1e',...
     '7  ', itnMM, xMM(1), norm(xMM), emptystr, errMM,  rnormMMout, rnormMM*beta1,...
      ArnormMMout, emptystr, emptystr, emptystr, tocMM));
    gprintf(sprintf('\n%s %6d %11.3e %11.3e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e',...
     '8  ', itn, x(1), norm(x), xnorm, err , rnormout, rnorm, Arnormout,...
     Arnorm, Anorm, Acond, tocm));
    gprintf(sprintf('\n%s %6d %11.3e %11.3e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e',...
     '9  ', itn1, x1(1), norm(x1), xnorm1, err1, rnorm1out, rnorm1, ...
     Arnorm1out, Arnorm1, Anorm1, Acond1, tocSOL));
    gprintf(sprintf('\n%s %6d %11.3e %11.3e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e',...
     '10 ', itn2, x2(1), norm(x2), xnorm2, err2, rnorm2out, rnorm2, ...
     Arnorm2out, Arnorm2, Anorm2, Acond2, tocQLP));

    gprintf(sprintf('\n%s %6d %11.3e %11.3e %s %10.1e %10.1e %10.1e %10.1e %s %s %s %10.1e',...
     '11 ', iterlm*2, xlm(1), norm(xlm), emptystr, errlm, rnormlmout, relreslm*beta1,...
      Arnormlmout, emptystr, emptystr, emptystr, toclm));   
     gprintf(sprintf('\n%s %6d %11.3e %11.3e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e',...
     '12 ', iterl*2, xl(1), norm(xl), xnorml, errl, rnormlout, rnorml,...
      Arnormlout, arnorml, anorml, acondl, tocl));

     gprintf(sprintf('\n%s %6d %11.3e %11.3e %s %10.1e %10.1e %10.1e %10.1e %s %s %s %10.1e',...
      '13 ', iterg, xg(1), norm(xg), emptystr, errg, rnormgout, relresg*beta1,...
       Arnormgout, emptystr, emptystr, emptystr, tocg));
     gprintf(sprintf('\n%s %6d %11.3e %11.3e %s %10.1e %10.1e %10.1e %10.1e %s %s %s %10.1e',...
      '14 ', itersq, xsq(1), norm(xsq), emptystr, errsq, rnormsqout, relressq*beta1,...
       Arnormsqout, emptystr, emptystr, emptystr, tocsq));
     gprintf(sprintf('\n%s %6d %11.3e %11.3e %s %10.1e %10.1e %10.1e %10.1e %s %s %s %10.1e',...
      '15 ', iterq*2, xq(1), norm(xq), emptystr, errq, rnormqout, relresq*beta1,...
       Arnormqout, emptystr, emptystr, emptystr, tocq));

       gprintf(sprintf('\n%s %6d %11.3e %11.3e %s %10.1e %10.1e %10.1e %10.1e %s %s %s %10.1e',...
      '16 ', iterb*2, xb(1), norm(xb), emptystr, errb, rnormbout, relresb*beta1,...
       Arnormbout, emptystr, emptystr, emptystr, tocb));
     gprintf(sprintf('\n%s %6d %11.3e %11.3e %s %10.1e %10.1e %10.1e %10.1e %s %s %s %10.1e',...
      '17 ', iterbc*2, xbc(1), norm(xbc), emptystr, errbc, rnormbcout, relresbc*beta1,...
       Arnormbcout, emptystr, emptystr, emptystr, tocbc));


    fprintf('\n');

    
 
timeiter = toc;
 
if exist('gamavecSOL', 'var')
   fignum = 1;
   len = length(gamavecQLP);
   if length(gamavecSOL) < length(gamavecQLP)
       len = length(gamavecSOL);
   end
   plotEigGamma(fignum, n, ts, gamavecQLP(1:len), gamavecSOL,  alfavecQLP(1:len), betavecQLP(1:len), alfavecSOL, betavecSOL, mu)
   if egNo == 2
          mfig = 2; 
          nfig = 3;
          h    = subplot(mfig,nfig,3);
          set(h, 'YLim',  [1e-4 1  ]);
          set(h, 'YTick', [1e-4  1e-2  1]);
          set(h, 'XLim',  [0 20]);
          set(h, 'XTick', [0: 10: 20]);
          h    = subplot(mfig,nfig,6);
          set(h, 'YLim',  [1e-8 1  ]);
          set(h, 'YTick', [1e-8  1e-4  1]);
          set(h, 'XLim',  [0 20]);
          set(h, 'XTick', [0: 10: 20]);
   end
   print( gcf, '-depsc2', ['Davis', num2str(id), '.eps'])
end

      
%% save files
save([filename,'results']) % .mat
diary off;