function test_minresqlp_fig7_1(probnum)

% FILE_NAME: test_minresqlp_fig7_1.m
%  
% USAGE:
%    test_minresqlp_fig7_1(probnum);
%
% INPUTS:
%    probnum  integer, example number: 1 or 2
%  
% OUTPUTS:
%    --     
%
% EXAMPLE:
%    test_minresqlp_fig7_1(2)
%
% SEE ALSO:
%    testminresQLP27
%
% REFERENCES:
%    (1) S.-C. Choi,
%        Iterative Methods for Singular Linear Equations and 
%        Least-Squares Problems.
%        PhD thesis, Stanford University, December 2006.
%    
%    (2) C. C. Paige and M. A. Saunders,
%        Solution of sparse indefinite systems of linear equations,
%        SIAM J. Numerical Analysis 12, 617-629 (1975). 
%
%    (3) S.-C. Choi, C. C. Paige and M. A. Saunders,
%        MINRES-QLP: A Krylov subspace method for indefinite or
%        singular symmetric systems,
%        SIAM J. Sci. Comput., submitted 07 Mar 2010.
%
% MODIFICATION HISTORY:
% 26 Jul 2006: Renamed testminresQLP27.m to test_minresqlp_fig7_1.m.
%              Cleaned up the script. 
%              Updated minresQLP43 to minresQLP49. 
%              Updated minresQLPs7 to minresQLP. 
% 22 Mar 2010: Current version.
%
% AUTHORS: Sou-Cheng (Terrya) Choi, ICME, Stanford University
%          Michael Saunders, SOL / ICME, Stanford University
%

close all
format compact
format short e

disp('==================== Set up input parameters ====================')

rand('state', 0);  

M        = [];
iw       = [];
rw       = [];
Acondlim = 1e14;
precond  = false;
shift    = 0;
show     = true;  
maxxnorm = 1e9;
rtol     = 100*eps;
debug    = 0;
isbestiter = 0;

errevd   = -1;  
errtevd  = -1;   
errMM    = -1;
err      = -1;
err1     = -1;
err2     = -1;
errpcg   = -1;
errpcgi  = -1;
errsym   = -1;
errMS    = -1;
errsq    = -1;
mu       = 1;
isdiag   = 0;
disable  = 0;
TranCond = 1e7;

switch probnum 
  case 1 % small example n = 25  % almost compatible
    load('Davis1177.mat')
    A        = Problem.A;
    n        = length(A);
    id       = Problem.id;
    title    = Problem.title;
    itnlim   = n;
    show     = true;
    shift    = 0;
    maxxnorm = 1e2;
    shift0   = 4;     %  -2;
    shiftn   = 4.001; % 5.5;
    shifth   = 0;
    mu       = 10;
    b        = (A - shift*speye(n))*rand(n,1);  % compatible 
    rtol     = 1e-11;

  case 2 % small example n = 25
    load('Davis1177.mat')
    A        = Problem.A;
    n        = length(A);
    id       = Problem.id;
    title    = Problem.title;
    itnlim   = n*2;
    show     = true;
    smallr   = 1e-5*rand(n,1);
    b        = (A - shift*speye(n))*rand(n,1) + smallr;  % almost compatible
    rtol     = 1e-14;
    shift    = 0;     % eigenvalue
    shift0   = 4;     %  -2;
    shiftn   = 4.001; % 5.5;
    shifth   = 0;
    maxxnorm = 1e4;  % 1e7
end


filename = ['Davis', num2str(id), 'Case', num2str(probnum), 'Eig'];
diary([filename,'.txt'])

fprintf('\nComputing EVD and TEVD of A and the solutions...')

[tx, trankA, tcondA, ts, xevd, rankA, condA, normA, ...
      errVD, terrVD, ortherrVD, tortherrVD, s, V, tsInd, snzInd] = tevd7(A,b,[],[],[],mu);

fprintf('\nSaving EVD and TEVD of A in %s...', filename)
save(filename, 's', 'V')
ss = s;
if shift ~= 0 
  ss = ss - shift;
end


xevdnorm = norm(xevd);
xsnorm   = norm(tx);
if shift ~= 0
  revd      = A*xevd - shift*xevd - b;
  rs        = A*tx   - shift*tx   - b;
  Arevdnorm = norm(A*revd - shift*revd);
  Arsnorm   = norm(A*rs   - shift*rs);
else
  revd      = A*xevd - b;
  rs        = A*tx   - b;
  Arevdnorm = norm(revd'*A);
  Arsnorm   = norm(rs'*A);
end  
revdnorm = norm(revd);
rsnorm = norm(rs);
if exist('xevd') 
  errtevd   = norm(tx - tx);
  errevd    = norm(tx - xevd);
  normA     = max(ts);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Print Inputs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

beta1 = norm(b);
nnzA  = nnz(A-shift*speye(n));
fprintf('\nMatrix ID = %10d,   title    = %s', id, title)
fprintf('\nn         = %10d,   maxit    = %10d,   nnz       = %10d,  nnz / n ~= %6d.',n, itnlim,nnzA,ceil( nnzA / n))
fprintf('\nshift     = %10.3e,   tol      = %10.0e,  maxxnorm   = %10.0e,  maxcond = %8.1e.', shift, rtol, maxxnorm, Acondlim)
fprintf('\n||b||     = %8.1e.', beta1)


if ~isequal(A,A') 
  warning('Symmetrizing A by (A+A'')/2...')
  A = (A+A')/2;
end


%% Iterative methods

fprintf('\n========================= minresSOL69 ========================')
fignum  = 0; %2; 
testsym = 1;
[x1, istopSOL, itn1, rnorm1, Arnorm1, Anorm1, Acond1, xnorm1, Axnorm,            ...
 resvec, Aresvec, xnormvec, pnormvecSOL, gamavecSOL, alfavecSOL, betavecSOL, VV, ...
 resvecdirect, relresvec, Aresvecdirect, relAresvec, pnormdirectvec, Axnormvec]  = ...
   minresSOL69( A, b, rtol, itnlim, M, shift, maxxnorm, Acondlim,  show, testsym,  ...
                fignum, debug, disable, isbestiter);
 
r1         = b - A*x1 + shift*x1;
rnorm1out  = norm(r1);
Arnorm1out = norm(A*r1 - shift*r1);
if exist('xevd')
  err1 = norm(tx - x1);
end
        
fprintf('\n========================= minresQLP49 ========================')
  
fignum     = 0;  % 3; 
[x2, istopQLP, itn2, rnorm2, Arnorm2, Anorm2, Acond2, xnorm2, Axnorm2,  ...
 resvec2, Aresvec2, xnormvec2, pnormvecQLP, gamavecQLP, alfavecQLP, betavecQLP, ...
 veplnvecQLP, etavecQLP, VVmatQLP, resvecdirectQLP, relresvecQLP,               ...
 AresvecdirectQLP, relAresvecQLP, xnormdirectvecQLP, pnormdirectvecQLP, Axnormvec2QLP] = ...
   minresQLP49( A, b, rtol, itnlim, M, shift, maxxnorm, Acondlim, show, ...
                testsym, fignum, debug, disable, isbestiter );
  
r2         = b - A*x2 +shift*x2;
rnorm2out  = norm(r2);
Arnorm2out = norm(A*r2 - shift*r2);
if exist('xevd')
  err2 = norm(tx - x2); 
end

 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Print Inputs again
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
nnzA = nnz(A - shift*speye(n));

if id > 0
  fprintf('\nMatrix ID = %4d, title  = %s', id, title)
else
  fprintf('\nTitle  = %s', id, title)
end
if isempty(Acondlim)
  Acondlim = 1e14;
end
fprintf('\nn     = %5d,   maxit = %5d,   nnz    = %5d,   nnz / n = %5d.', n, itnlim, nnzA, ceil(nnzA/n))
gprintf(sprintf('\nshift = %5.2e,  tol = %5.0e, maxxnorm = %5.0e,  maxcond = %5.0e, ||b|| = %5.1e. ', ...
                shift, rtol, maxxnorm, Acondlim, beta1));  

fprintf('\n========================= Results ============================')
fprintf('\nMethods:')
emptystr = '   --    ';
emptstr2 = '    --';

if exist('xevd')
  fprintf('\n1 : EVD.               ');    gprintf(sprintf('|| AV - VD ||_2 / ||A||_2 = %10.2e.', errVD/normA));
  fprintf('\n                       ');    gprintf(sprintf('|| V''V - I ||_2 = %10.2e', ortherrVD));
  fprintf('\n2 : TEVD.              ');    gprintf(sprintf('|| AV - VD ||_2 / ||A||_2 = %10.2e.', terrVD/normA));
  fprintf('\n                       ');    gprintf(sprintf('|| V''V - I ||_2 = %10.2e', tortherrVD));
end
 
fprintf('\n3: MINRES SOL69.       ');   methodno = 3;  printflagmsg(methodno,istopSOL);
fprintf('\n4: MINRES QLP49.       ');   methodno = 4;  printflagmsg(methodno,istopQLP); 
fprintf('\n')
fprintf(['\nMethod A*v  x(1)       ||x||               ||e||      ', ...
         '||r||               ||Ar||              ||A||      K(A)     '])
fprintf(['\n                       direct   recurrent =||x-xTEVD|| direct   ', ...
         'recurrent  direct  recurrent       \n'])
emptystr = '   --';
emptystr2 = '    --   ';
if exist('xevd')
  gprintf(sprintf('\n%s %s %11.3e %11.3e %s %10.1e %10.1e %s %10.1e %s %10.1e %10.1e ', ...
                  '1  ', emptystr, xevd(1), xevdnorm, emptystr2, errevd, revdnorm, emptystr2, ...
                  Arevdnorm, emptystr2, normA, condA));
  gprintf(sprintf('\n%s %s %11.3e %11.3e %s %10.1e %10.1e %s %10.1e %s %10.1e %10.1e ', ...
                  '2  ', emptystr, tx(1), xsnorm, emptystr2, errtevd, rsnorm, emptystr2, ...
                  Arsnorm, emptystr2, normA, tcondA));
end
   
gprintf(sprintf('\n%s %6d %11.3e %11.3e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e', ...
                '3 ', itn1, x1(1), norm(x1), xnorm1, err1, rnorm1out, rnorm1, ...
                Arnorm1out, Arnorm1, Anorm1, Acond1));
gprintf(sprintf('\n%s %6d %11.3e %11.3e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e %10.1e', ...
                '4 ', itn2, x2(1), norm(x2), xnorm2, err2, rnorm2out, rnorm2, ...
                Arnorm2out, Arnorm2, Anorm2, Acond2));

fprintf('\n')
options={'Interpreter','latex'};

if exist('gamavecSOL')
  fignum = 1;
  figure(fignum);
  plotEigGamma(fignum, n, ts, gamavecQLP, gamavecSOL,  alfavecQLP, betavecQLP, alfavecSOL, betavecSOL, mu)
  if probnum == 2
    mfig = 2; 
    nfig = 3;
    h    = subplot(mfig,nfig,3);
    set(h, 'YLim',  [1e-4 1  ]);
    set(h, 'YTick', [1e-4  1e-2  1]);
    set(h, 'XLim',  [1 19]);
    h    = subplot(mfig,nfig,6);
    set(h, 'YLim',  [1e-8 1  ]);
    set(h, 'YTick', [1e-8  1e-4  1]);
    set(h, 'XLim',  [1 20]);
  end
  print( gcf, '-depsc2', ['Davis', num2str(id), '.eps'])
end

if (itn1 > 2) & (itn2 > 2)
  fignum = 2;
  h      = figure(fignum);
  mfig   = 1;
  nfig   = 3;
  set(h, 'position', [288    24   885   910]);
  subplot(mfig,nfig,1);
 
  semilogy(1:itn1, max(resvec, realmin), '-b', 'linewidth', 2  );     hold on; 
  semilogy(1:itn2, max(resvec2, realmin), '^r', 'linewidth', 1.5  ); 
  semilogy(1:itn1, max(relresvec, realmin), 'b.'); 
  semilogy(1:itn2, max(relresvecQLP, realmin), 'ro'); 
  semilogy([1; max(itn1,itn2)], rtol * ones(2,1), ':g');   hold off;   
  h = legend('$\phi_k^M$','$\phi_k^Q$', 'NRBE $\phi_k^M$', 'NRBE $\phi_k^Q$');
  set(h, options{:}, 'fontsize',9); 
  hold off
  %ylabel('$\phi_k$', options{:}); 
  xlabel('$k$', options{:});  axis square;  % axis tight 
    
  subplot(mfig,nfig,2);
  semilogy(1:itn1, max(Aresvec, realmin), '-b', 'linewidth', 2  );   hold on;  
  semilogy(1:itn2, max(Aresvec2, realmin), '^r', 'linewidth', 1.5  );   
  semilogy(1:itn1, max(relAresvec , realmin), 'b.'); 
  semilogy(1:itn2, max(relAresvecQLP, realmin), 'ro'); 
  semilogy([1; max(itn1,itn2)], rtol * ones(2,1), ':g'); hold off;  
  h = legend('$\psi_{\phantom{1}k}$', '$\|Ar_k\|$','$\psi_{\phantom{1}k}/\|A\|$','tol');
  hold off
  %ylabel('$\psi_{\phantom{1}k}$', options{:}); % actually phi_k, check figure(1), 2nd plot, differences small
  h = legend('$\psi_{\phantom{1}k}^{\phantom{1}M}$','$\psi_{\phantom{1}k}^{\phantom{1}Q}$', ...
             'NRBE $\psi_{\phantom{1}k}^{\phantom{1}M}$', 'NRBE $\psi_{\phantom{1}k}^{\phantom{1}Q}$');
  set(h,  options{:}, 'fontsize',9);
  xlabel('$k$', options{:});  axis square; %axis tight 
    
  subplot(mfig,nfig,3);
 
  semilogy(1:itn1, max(xnormvec, realmin), '-b', 'linewidth', 2 );   hold on; 
  semilogy(1:itn2, max(xnormvec2, realmin), '^r', 'linewidth', 1.5  );     
  semilogy([1; itn2], maxxnorm * ones(2,1), ':g');   hold off;
  %ylabel('$\chi_k$',options{:}); % actually phi_k, check figure(1), 2nd plot, differences small
  h = legend('$\|x_k^M\|$','$\chi_k^Q$');
  set(h, options{:},'fontsize',9);
  xlabel('$k$', options{:});  axis square; %axis tight 
    
  if probnum == 2
    h = subplot(mfig,nfig,1);
    set(h, 'YLim',  [1e-15 1e1]);
    set(h, 'YTick', [1e-14 1e-7 1e0]);
    set(h, 'XLim',  [1 21]);
    set(h, 'Position', [1.3000e-001  1.1000e-001  2.0148e-001  8.1500e-001]);
   %set(h, 'Position', [1.3000e-001  1.1000e-001  2.0148e-001  8.1500e-001]);
    h = subplot(mfig,nfig,2);
    set(h, 'YLim',  [1e-15 1e1]);
    set(h, 'YTick', [1e-14 1e-7 1e0]);
    set(h, 'XLim',  [1 21]);
    set(h, 'Position', [4.2911e-001  1.1000e-001 2.0148e-001  8.1500e-001]); 
    h = subplot(mfig,nfig,3);
    set(h, 'YLim',  [0 1e3]);
    set(h, 'YTick', [0 1e1 1e2]);
    set(h, 'XLim',  [1 21]);
    set(h, 'Position', [7.0352e-001  1.1000e-001  2.0148e-001  8.1500e-001]);
  end
end


%% save files
for i = 1:2
  figure(i) 
  print( gcf, '-depsc2', [filename, num2str(i)]) % save .eps
  print( gcf, '-dpdf',   [filename, num2str(i)])  % save .pdf
  hgsave([filename, num2str(i),'results']) % save .fig
end
save([filename,'results'])  % .mat
diary off
% keyboard



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% private function plotEigGamma
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function plotEigGamma(figno, n, ts, gamavecQLP, gamavecSOL, alfavecQLP, betavecQLP, alfavecSOL, betavecSOL, mu)

figure(figno)
tol = eps * max(abs([ts(1), ts(end)])) * mu;
 
TQLP       = diag(betavecQLP(1:end-1),1) + diag(alfavecQLP,0) + diag(betavecQLP(1:end-1),-1);
absEigTQLP = abs(eig(TQLP));

TSOL       = diag(betavecSOL(1:end-1),1) + diag(alfavecSOL,0) + diag(betavecSOL(1:end-1),-1);
absEigTSOL = abs(eig(TSOL));
 
mfig = 2;
nfig = 3;
ts   = flipud(sort(abs(ts)));
subplot(mfig,nfig,1) 
plot(ts,'b.');  title(['nonzero \sigma(A) = |\lambda(A)|, n = ', num2str(n)])
axis square; axis tight

subplot(mfig,nfig,5) 
[x, err] = eigFit(ts, gamavecQLP, 'b.', 'rO');
title(['QLP\gamma_k, k = ', num2str(length(gamavecQLP))]);
axis square; axis tight
 
subplot(mfig,nfig,6)
semilogy(x, abs(err), 'r.'); title('| QLP\gamma_k - \sigma(A) |');
axis square; axis tight

subplot(mfig,nfig,2)
[x, err] = eigFit(ts, gamavecSOL, 'b.', 'rO');
title(['SOL\gamma_k, k = ', num2str(length(gamavecSOL))]);
axis square; axis tight

subplot(mfig,nfig,3) 
semilogy(x, abs(err), 'r.'); title('| SOL\gamma_k - \sigma(A) |');
if n==25, v = axis;  v(2) = 25;  v(3) = 1e-8;  axis(v); end
axis square; axis tight    
